<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\VendorMessages\Model\Repo;

use XCart\Extender\Mapping\Extender;

/**
 * Messages repository
 *
 * @Extender\Mixin
 * @Extender\After ("XC\VendorMessages")
 * @Extender\Depend ("XC\MultiVendor")
 */
class MessageMultivendor extends \XC\VendorMessages\Model\Repo\Message
{
    /**
     * Count unread messages (for admin)
     *
     * @param \XLite\Model\Profile|null $profile Profile
     *
     * @return integer
     */
    public function countUnreadForAdmin(\XLite\Model\Profile $profile = null)
    {
        $profile = $profile ?: \XLite\Core\Auth::getInstance()->getProfile();

        return count($this->defineCountUnreadForAdminQuery($profile)->getArrayResult());
    }

    /**
     * Count unread messages (for vendor)
     *
     * @param \XLite\Model\Profile|null $profile Profile
     *
     * @return integer
     */
    public function countUnreadForVendor(\XLite\Model\Profile $profile = null)
    {
        $profile = $profile ?: \XLite\Core\Auth::getInstance()->getProfile();

        return count($this->defineCountUnreadForVendorQuery($profile)->getArrayResult());
    }

    /**
     * Count vendor's messages
     *
     * @param \XLite\Model\Profile $profile Vendor profile OPTIONAL
     *
     * @return integer
     */
    public function countByVendor(\XLite\Model\Profile $profile = null)
    {
        $profile = $profile ?: \XLite\Core\Auth::getInstance()->getProfile();

        return $this->defineCountByVendorQuery($profile)->count();
    }

    /**
     * Count opened disputes
     *
     * @return integer
     */
    public function countDisputes()
    {
        return count($this->defineCountDisputesQuery()->getArrayResult());
    }

    /**
     * Find last messages which open dispute
     *
     * @param \XLite\Model\Order $order Order
     *
     * @return \XC\VendorMessages\Model\Message
     */
    public function findOneLastOpenDispute(\XLite\Model\Order $order)
    {
        return $this->defineFindOneLastOpenDisputeQuery($order)->getSingleResult();
    }

    /**
     * Define query for 'countUnreadForAdmin' method
     *
     * @param \XLite\Model\Profile $profile Profile
     *
     * @return \XLite\Model\QueryBuilder\AQueryBuilder
     */
    protected function defineCountUnreadForAdminQuery(\XLite\Model\Profile $profile)
    {
        $qb = $this->createQueryBuilder('m')
            ->select('m.id')
            ->linkLeft('m.readers', 'r0', \Doctrine\ORM\Query\Expr\Join::WITH, 'r0.reader = :reader')
            ->linkLeft('m.readers', 'r1')
            ->linkInner('m.conversation', 'conv')
            ->groupBy('m.id')
            ->andHaving('COUNT(r1.id) != SUM(IFELSE(r0.id IS NULL, 0, 1)) OR COUNT(r1.id) = 0')
            ->setParameter('reader', $profile);

        return $qb;
    }

    /**
     * Define query for 'countDisputes' method
     *
     * @return \XLite\Model\QueryBuilder\AQueryBuilder
     */
    protected function defineCountDisputesQuery()
    {
        return $this->createQueryBuilder('m')
            ->select('o.order_id')
            ->linkInner('m.conversation', 'conv')
            ->linkInner('conv.order', 'o')
            ->andWhere('o.is_opened_dispute = :enabled_dispute')
            ->setParameter('enabled_dispute', true)
            ->groupBy('o.order_id');
    }

    /**
     * Define query for 'countUnreadForVendor' method
     *
     * @param \XLite\Model\Profile $profile Profile
     *
     * @return \XLite\Model\QueryBuilder\AQueryBuilder
     */
    protected function defineCountUnreadForVendorQuery(\XLite\Model\Profile $profile)
    {
        return $this->createQueryBuilder('m')
            ->select('m.id')
            ->linkLeft('m.readers', 'r0', \Doctrine\ORM\Query\Expr\Join::WITH, 'r0.reader = :reader')
            ->linkLeft('m.readers', 'r1')
            ->linkInner('m.conversation', 'conv')
            ->andWhere(':vendor MEMBER OF conv.members')
            ->groupBy('m.id')
            ->andHaving('COUNT(r1.id) != SUM(IFELSE(r0.id IS NULL, 0, 1)) OR COUNT(r1.id) = 0')
            ->setParameter('reader', $profile)
            ->setParameter('vendor', $profile);
    }

    /**
     * Define query for 'countByVendor' method
     *
     * @param \XLite\Model\Profile $profile Profile
     *
     * @return \XLite\Model\QueryBuilder\AQueryBuilder
     */
    protected function defineCountByVendorQuery(\XLite\Model\Profile $profile)
    {
        return $this->createQueryBuilder('m')
            ->linkInner('m.conversation', 'conv')
            ->andWhere(':vendor MEMBER OF conv.members')
            ->setParameter('vendor', $profile);
    }

    /**
     * Define query for 'findOneLastOpenDispute' method
     *
     * @param \XLite\Model\Order $order Order
     *
     * @return \XLite\Model\QueryBuilder\AQueryBuilder
     */
    protected function defineFindOneLastOpenDisputeQuery(\XLite\Model\Order $order)
    {
        return $this->createQueryBuilder('m')
            ->linkInner('m.conversation', 'conv')
            ->andWhere('conv.order = :order AND m.type = :dispute_opener')
            ->setParameter('order', $order)
            ->setParameter('dispute_opener', \XC\VendorMessages\Model\Message::MESSAGE_TYPE_DISPUTE_OPEN)
            ->orderBy('m.date', 'DESC')
            ->setMaxResults(1);
    }
}
