<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XC\Upselling\Core\EventListener;

use Doctrine\DBAL\Exception;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Event\LoadClassMetadataEventArgs;

class UniqueConstraintsEventListener
{
    /**
     * @param LoadClassMetadataEventArgs $eventArgs
     *
     * @throws Exception
     */
    public function loadClassMetadata(LoadClassMetadataEventArgs $eventArgs): void
    {
        $classMetadata = $eventArgs->getClassMetadata();

        if (
            $classMetadata->name === \XC\Upselling\Model\UpsellingProduct::class
            && !isset($classMetadata->table['uniqueConstraints']['parent_child_product_unique'])
            && !isset($classMetadata->table['uniqueConstraints']['parent_child_product'])
        ) {
            // changes in data
            $this->removeDuplicates($eventArgs->getEntityManager(), $classMetadata->table['name']);

            // changes in structure
            $result = [];
            foreach (['indexes', 'uniqueConstraints'] as $indexType) {
                $result[$indexType] = $classMetadata->table[$indexType] ?? [];
                unset($result[$indexType]['parent_child_product']);
            }

            if (!isset($result['uniqueConstraints']['parent_child_product'])) {
                $result['uniqueConstraints'] = array_merge($result['uniqueConstraints'], ['parent_child_product' => ['columns' => ['parent_product_id', 'product_id']]]);
            }

            $classMetadata->setPrimaryTable($result);
        }
    }

    /**
     * @param EntityManagerInterface $em
     * @param string                 $tableName
     *
     * @throws Exception
     */
    private function removeDuplicates(\Doctrine\ORM\EntityManagerInterface $em, string $tableName): void
    {
        $dbal          = $em->getConnection();
        $schemaManager = $dbal?->createSchemaManager();
        if (!in_array($tableName, $schemaManager?->listTableNames() ?: [], true)) {
            return; // no table - no problems
        }

        // Test to avoid running complex query on huge DBs
        $tableIndexes = $schemaManager?->listTableIndexes($tableName) ?: [];
        foreach ($tableIndexes as $tableIndex) {
            if (
                in_array($tableIndex->getName(), ['parent_child_product', 'parent_child_product_unique'])
                && $tableIndex->isUnique()
                && count($tableIndex->getColumns()) > 1
            ) {
                // mysql guarantees that there are no duplicates
                return;
            }
        }

        // Attn! DB specific query,
        // probably it is better to use some abstraction over a DB engine or implement the `query` for each supported storage
        $relatedProductIds = $dbal->fetchAllAssociative("
            SELECT DISTINCT t2.`id` FROM {$tableName} t1 INNER JOIN {$tableName} t2 
             ON t1.`product_id`=t2.`product_id` AND t1.`parent_product_id`=t2.`parent_product_id` WHERE t1.`id` != t2.`id`") ?: [];

        array_pop($relatedProductIds); // keep at least one row
        if (!$relatedProductIds) {
            return; // we are clean
        }

        foreach ($relatedProductIds as $relatedProduct) {
            $dbal->delete($tableName, ['id' => $relatedProduct['id']]);
        }
    }
}
