<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\UPS\Model\Shipping\Mapper\Rate;

use XC\UPS\Model\Shipping;
use XLite\Core;

class InputMapper extends Shipping\Mapper\AMapper
{
    protected $weightUnit;

    protected $dimensionUnit;

    protected $destinationCurrency;

    /**
     * Is mapper able to map?
     *
     * @return boolean
     */
    protected function isApplicable()
    {
        return $this->inputData && is_array($this->inputData);
    }

    protected function performMap(): string
    {
        $srcAddress = $this->inputData['srcAddress'];
        if ($srcAddress['country'] === 'US' && isset($srcAddress['state']) && $srcAddress['state'] === 'PR') {
            $srcAddress['country'] = 'PR';
            $srcAddress['state'] = '';
            $this->inputData['srcAddress'] = $srcAddress;
        }

        if (in_array($srcAddress['country'], ['DO', 'PR', 'US', 'CA'], true)) {
            $this->weightUnit = 'LBS';
            $this->dimensionUnit = 'IN';
        } else {
            $this->weightUnit = 'KGS';
            $this->dimensionUnit = 'CM';
        }

        $config = $this->getConfiguration();
        $dstAddress = $this->inputData['dstAddress'];
        if ($dstAddress['country'] === 'US' && isset($dstAddress['state']) && $dstAddress['state'] === 'PR') {
            $dstAddress['country'] = 'PR';
            $dstAddress['state'] = '';
            $this->inputData['dstAddress'] = $dstAddress;
        }

        $dstCountry = Core\Database::getRepo('XLite\Model\Country')
            ->findOneBy(['code' => $dstAddress['country']]);
        $this->destinationCurrency = $dstCountry && $dstCountry->getCurrency()
            ? $dstCountry->getCurrency()->getCode()
            : $config->currency_code;

        return json_encode($this->getRatingServiceSelectionRequest());
    }

    protected function getRatingServiceSelectionRequest(): array
    {
        $config = $this->getConfiguration();

        $result['RateRequest'] = [];
        $result['RateRequest']['Request']['RequestOption'] = 'Rate';
        $result['RateRequest']['PickupType']['Code'] = $config->pickup_type;
        $result['RateRequest']['Shipment'] = $this->getShipment();

        if ($customerClassification = $this->getCustomerClassification()) {
            $result['RateRequest']['CustomerClassification'] = $customerClassification;
        }

        return $result;
    }

    protected function getCustomerClassification(): array
    {
        $config = $this->getConfiguration();

        $srcCountry = $this->inputData['srcAddress']['country'];
        // return Rate associated with account number if negotiated rates are requested
        if (
            $srcCountry === 'US'
            && $this->isNegotiatedShipperNumber($config)
        ) {
            $result['Code'] = '00';
        }

        return $result ?? [];
    }

    protected function getShipment(): array
    {
        $result = [];
        $result['Shipper'] = $this->getShipper();
        $result['ShipTo'] = $this->getShipTo();
        $result['ShipFrom'] = $this->getShipFrom();
        $result['Package'] = $this->getPackages();

        if ($shipmentServiceOptions = $this->getShipmentServiceOptions()) {
            $result['ShipmentServiceOptions'] = $shipmentServiceOptions;
        }

        $result['RateInformation'] = $this->getRateInformation();

        return $result;
    }

    protected function getShipper(): array
    {
        $result = [];
        $config = $this->getConfiguration();
        $srcAddress = $this->inputData['srcAddress'];

        if ($this->isNegotiatedShipperNumber($config)) {
            $result['ShipperNumber'] = $config->shipper_number;
        }

        $result['Address']['City'] = $srcAddress['city'];
        $result['Address']['StateProvinceCode'] = $srcAddress['state'];
        $result['Address']['PostalCode'] = $srcAddress['zipcode'];
        $result['Address']['CountryCode'] = $srcAddress['country'];

        return $result;
    }

    protected function getShipTo(): array
    {
        $result = [];
        $dstAddress = $this->inputData['dstAddress'];

        if (isset($dstAddress['type']) && $dstAddress['type'] === 'R') {
            $result['ResidentialAddressIndicator'] = true;
        }

        $result['Address']['City'] = $dstAddress['city'];
        $result['Address']['StateProvinceCode'] = $dstAddress['state'];
        $result['Address']['PostalCode'] = $dstAddress['zipcode'];
        $result['Address']['CountryCode'] = $dstAddress['country'];

        return $result;
    }

    protected function getShipFrom(): array
    {
        $result = [];
        $srcAddress = $this->inputData['srcAddress'];

        $result['Address']['City'] = $srcAddress['city'];
        $result['Address']['StateProvinceCode'] = $srcAddress['state'];
        $result['Address']['PostalCode'] = $srcAddress['zipcode'];
        $result['Address']['CountryCode'] = $srcAddress['country'];

        return $result;
    }

    protected function getPackages(): array
    {
        $result = [];
        foreach ($this->inputData['packages'] as $package) {
            $result[] = $this->getPackage($package);
        }

        return $result;
    }

    protected function getPackage(array $package): array
    {
        $result = [];
        $config = $this->getConfiguration();

        if ($serviceOptions = $this->getPackageServiceOptions($package)) {
            $result['PackageServiceOptions'] = $serviceOptions;
        }

        if ($config->additional_handling) {
            $result['AdditionalHandling'] = true;
        }

        $result['PackagingType']['Code'] = $config->packaging_type;
        $result['PackageWeight']['UnitOfMeasurement']['Code'] = $this->weightUnit;
        $result['PackageWeight']['Weight'] = (string) $package['weight'];
        $result['Dimensions'] = $this->getPackageDimensions($package);
        if ($largePackageIndicator = $this->getLargePackageIndicator($package)) {
            $result['LargePackageIndicator'] = $largePackageIndicator;
        }

        return $result;
    }

    protected function getPackageDimensions(array $package): array
    {
        $result = [];
        $config = $this->getConfiguration();

        if (isset($package['box'])) {
            $length = $package['box']['length'];
            $width = $package['box']['width'];
            $height = $package['box']['height'];
        } else {
            [$length, $width, $height] = $config->dimensions;
        }

        if ($length + $width + $height > 0) {
            $length = round($length, 2);
            $width = round($width, 2);
            $height = round($height, 2);

            $result['UnitOfMeasurement']['Code'] = $this->dimensionUnit;
            $result['Length'] = (string) $length;
            $result['Width'] = (string) $width;
            $result['Height'] = (string) $height;
        }

        return $result;
    }

    protected function getLargePackageIndicator($package): bool
    {
        $result = false;
        $config = $this->getConfiguration();

        if (isset($package['box'])) {
            $length = $package['box']['length'];
            $width = $package['box']['width'];
            $height = $package['box']['height'];
        } else {
            [$length, $width, $height] = $config->dimensions;
        }

        if ($length + $width + $height > 0) {
            $length = round($length, 2);
            $width = round($width, 2);
            $height = round($height, 2);

            $girth = $length + (2 * $width) + (2 * $height);
            if ($girth > 165) {
                $result = true;
            }
        }

        return $result;
    }

    protected function getPackageServiceOptions(array $package): array
    {
        $result = [];
        $config = $this->getConfiguration();
        $packageSubtotal = round((float) $package['subtotal'], 2);

        if ($config->extra_cover) {
            $insuredValue = round((float) $config->extra_cover_value, 2) ?: $packageSubtotal;
            if ($insuredValue > 0.1) {
                $result['InsuredValue']['CurrencyCode'] = $config->currency_code;
                $result['InsuredValue']['MonetaryValue'] = $insuredValue;
            }
        }

        $DCISType = (int) $config->delivery_conf;
        $srcCountry = $this->inputData['srcAddress']['country'];
        $dstCountry = $this->inputData['dstAddress']['country'];
        if (
            $DCISType > 0
            && $DCISType < 4
            && $srcCountry === 'US'
            && $dstCountry === 'US'
        ) {
            $result['DeliveryConfirmation']['DCISType'] = (string) $DCISType;
        }

        return $result;
    }

    protected function getShipmentServiceOptions(): array
    {
        $result = [];
        $config = $this->getConfiguration();

        if ($config->saturday_pickup) {
            $result['SaturdayPickupIndicator'] = true;
        }

        if ($config->saturday_delivery) {
            $result['SaturdayDeliveryIndicator'] = true;
        }

        return $result;
    }

    protected function getRateInformation(): array
    {
        $result = [];
        $config = $this->getConfiguration();

        if ($config->negotiated_rates) {
            $result['NegotiatedRatesIndicator'] = true;
        }

        return $result;
    }

    /**
     * Post-process mapped data
     *
     * @param mixed $mapped mapped data to post-process
     *
     * @return mixed
     */
    protected function postProcessMapped($mapped)
    {
        return $mapped;
    }

    /**
     * Is needed only for negotiated rates
     * https://developer.ups.com/api/reference?loc=en_US#operation/Rate!path=RateRequest/Shipment/Shipper/ShipperNumber&t=request
     * https://developer.ups.com/api/reference?loc=en_US#operation/Rate!path=RateRequest/CustomerClassification/Code&t=request
     */
    protected function isNegotiatedShipperNumber(Core\ConfigCell $config): bool
    {
        return $config->shipper_number && $config->negotiated_rates;
    }
}
