<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\UPS\Model\Shipping\API;

use XC\UPS\Model\Shipping\Mapper;
use XC\UPS\Model\Shipping\Processor;
use XLite\InjectLoggerTrait;

class UPS
{
    use InjectLoggerTrait;

    public const API_VERSION = 'v1';

    /**
     * @var Processor\UPS
     */
    protected $processor;

    /**
     * @param Processor\UPS $processor
     */
    public function __construct($processor)
    {
        $this->processor = $processor;
    }

    protected function getApiURL(): string
    {
        return $this->processor->isTestMode()
            ? 'https://wwwcie.ups.com/'
            : 'https://onlinetools.ups.com/';
    }

    protected function getApiURLShop(): string
    {
        return $this->getApiURL() . 'api/rating/' . static::API_VERSION . '/Shop';
    }

    protected function getApiURLRate(): string
    {
        return $this->getApiURL() . 'api/rating/' . static::API_VERSION . '/Rate';
    }

    /**
     * @param array $inputData
     *
     * @return mixed
     */
    public function getRates($inputData)
    {
        $url = $this->getApiURLShop();

        $request = new Request\JSONRequest($url, $inputData);
        $request->setInputMapper(new Mapper\Rate\InputMapper($this->processor));
        $request->setOutputMapper(new Mapper\Rate\OutputMapper($this->processor));
        $request->setCurrentConfiguration($this->processor);
        $request->sendRequest();

        $this->processor->addApiCommunicationMessage(
            [
                'method' => __METHOD__,
                'URL' => $url,
                'request' => $request->getRawRequest(),
                'response' => $request->getRawResponse(),
            ]
        );

        $this->getLogger('XC-UPS')->debug('', [
            'method' => __METHOD__,
            'URL' => $url,
            'request' => $request->getRawRequest(),
            'response' => $request->getRawResponse(),
        ]);

        return $request->getResponse();
    }

    /**
     * @param array $inputData
     *
     * @return mixed
     */
    public function getRatesCOD($inputData)
    {
        $url = $this->getApiURLShop();

        $request = new Request\JSONRequest($url, $inputData);
        $request->setInputMapper(new Mapper\RateCOD\InputMapper($this->processor));
        $request->setOutputMapper(new Mapper\RateCOD\OutputMapper($this->processor));
        $request->sendRequest();

        $this->processor->addApiCommunicationMessage(
            [
                'method' => __METHOD__,
                'URL' => $url,
                'request' => $request->getRawRequest(),
                'response' => $request->getRawResponse(),
            ]
        );

        $this->getLogger('XC-UPS')->debug('', [
            'method' => __METHOD__,
            'URL' => $url,
            'request' => \XLite\Core\XML::getInstance()->getFormattedXML($request->getRawRequest()),
            'response' => \XLite\Core\XML::getInstance()->getFormattedXML($request->getRawResponse()),
        ]);

        return $request->getResponse();
    }
}
