<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\UPS\Model\Shipping\API\Request;

use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use XC\UPS\Core\UpsApiService;
use XC\UPS\Model\Shipping\Mapper;
use XLite\InjectLoggerTrait;

class ARequest
{
    use InjectLoggerTrait;

    public const POST = 'POST';
    public const GET  = 'GET';
    public const PUT  = 'PUT';

    public const CONTENT_TYPE = 'text/plain';

    /**
     * @var string
     */
    protected $url;

    /**
     * @var array
     */
    protected $inputData;

    /**
     * @var string
     */
    protected $httpType = self::POST;

    /**
     * @var Mapper\IMapper
     */
    protected $inputMapper;

    /**
     * @var Mapper\IMapper
     */
    protected $outputMapper;

    protected string $rawResponse = '';

    protected array $currentConfiguration = [];

    /**
     * @param string $url
     * @param array  $inputData
     */
    public function __construct($url, $inputData)
    {
        $this->url       = $url;
        $this->inputData = $inputData;
    }

    public function setCurrentConfiguration(\XC\UPS\Model\Shipping\Processor\UPS $processor): void
    {
        $this->currentConfiguration = [];
    }

    public function setInputMapper(Mapper\IMapper $mapper)
    {
        $this->inputMapper = $mapper;
    }

    public function setOutputMapper(Mapper\IMapper $mapper)
    {
        $this->outputMapper = $mapper;
    }

    public function sendRequest($forceRenewAccessToken = false): void
    {
        try {
            $response = $this->getHttpClient()->request(
                $this->httpType,
                $this->url,
                [
                    'body'    => $this->prepareParameters(),
                    'headers' => [
                        'Authorization' => 'Bearer ' . UpsApiService::getInstance()->getAccessToken($this->currentConfiguration, $forceRenewAccessToken),
                        'Content-Type'  => static::CONTENT_TYPE,
                    ],
                ]
            );

            $this->rawResponse = $response->getContent(false);

            if (
                !$forceRenewAccessToken
                && $this->rawResponse
                && is_null($this->getResponse()) // must be called before $this->outputMapper->getErrorCodes()
                && in_array('250002', $this->outputMapper->getErrorCodes()) // {"response":{"errors":[{"code":"250002","message":"Invalid Authentication Information."}]}}
            ) {
                $isAccessTokenProblem = true;
            }
        } catch (\Throwable $e) {
            $this->getLogger('XC\UPS')->debug("Fail request to {$this->url} Reason: {$e->getMessage()} Thrown at {$e->getFile()}:{$e->getLine()}");
            $accessTokenNotFoundErrorMsg = '404 Not Found returned for "' . \XC\UPS\Core\ProxyApi::PROXY_HOST . \XC\UPS\Core\ProxyApi::ACCESS_TOKEN_ENDPOINT . '"';
            $isAccessTokenProblem        = stripos($e->getMessage(), $accessTokenNotFoundErrorMsg) !== false;
        }

        if (
            !$forceRenewAccessToken
            && !empty($isAccessTokenProblem)
        ) {
            $forceRenewAccessToken = true;
            $this->sendRequest($forceRenewAccessToken);
        }
    }

    /**
     * @return mixed
     */
    public function getResponse()
    {
        static $res = [];
        if ($this->outputMapper) {
            $rawResponse    = $this->getRawResponse();
            $rawResponseKey = md5($rawResponse);
            if (array_key_exists($rawResponse, $res)) {
                return $res[$rawResponseKey];
            }
            $this->outputMapper->setInputData($rawResponse);
            $this->outputMapper->setInputData($this->inputData, 'request');
            $result = $res[$rawResponseKey] = $this->outputMapper->getMapped();
        } else {
            $result = $this->getRawResponse();
        }

        return $result;
    }

    /**
     * @return mixed
     */
    public function getRawRequest()
    {
        return $this->prepareParameters();
    }

    /**
     * @return string
     */
    public function getRawResponse()
    {
        return $this->rawResponse;
    }

    protected function getHttpClient(): HttpClientInterface
    {
        return HttpClient::create();
    }

    /**
     * @return string
     */
    protected function prepareParameters()
    {
        $result = '';

        if ($this->inputMapper) {
            $this->inputMapper->setInputData($this->inputData);
            $result = $this->inputMapper->getMapped();
        }

        return $result;
    }
}
