<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\UPS\EntityCallback;

use Doctrine\ORM\Event\PostLoadEventArgs;
use Doctrine\Persistence\Event\LifecycleEventArgs;
use XC\UPS\Core\Crypter;
use XLite\Model\Config;

final class ConfigCallback
{
    public function __construct(
        private Crypter $crypter
    ) {
    }

    public function prePersist(Config $config, LifecycleEventArgs $args): void
    {
        if (!$this->isExpectedConfig($config)) {
            return;
        }

        $encryptedToken = $this->encrypt($config->getValue());
        $config->setValue($encryptedToken);
    }

    public function preUpdate(Config $config, LifecycleEventArgs $args): void
    {
        if (!$this->isExpectedConfig($config)) {
            return;
        }

        $encryptedToken = $this->encrypt($config->getValue());
        $config->setValue($encryptedToken);
    }

    public function postPersist(Config $config, LifecycleEventArgs $args): void
    {
        if (!$this->isExpectedConfig($config)) {
            return;
        }

        $encryptedToken = $this->decrypt($config->getValue());
        $config->setValue($encryptedToken);
    }

    public function postUpdate(Config $config, LifecycleEventArgs $args): void
    {
        if (!$this->isExpectedConfig($config)) {
            return;
        }

        $encryptedToken = $this->decrypt($config->getValue());
        $config->setValue($encryptedToken);
    }

    public function postLoad(Config $config, PostLoadEventArgs $args): void
    {
        if (!$this->isExpectedConfig($config)) {
            return;
        }

        $encryptedToken = $this->decrypt($config->getValue());
        $config->setValue($encryptedToken);
    }

    private function encrypt(string $token): string
    {
        return $this->crypter->encrypt($token);
    }

    private function decrypt(string $token): string
    {
        return $this->crypter->decrypt($token);
    }

    private function isExpectedConfig(Config $config): bool
    {
        return $config->getCategory() === 'XC\UPS'
            && (
                $config->getName() === 'refresh_token'
                || $config->getName() === 'refresh_token_test'
            )
            && $config->getValue() !== '';
    }
}
