<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XC\UPS\Core;

use XLite\Base\Singleton;
use XLite\Core\Database;
use XLite\Model\Config;
use XLite\Model\Profile;

class UpsApiService extends Singleton
{
    public const BUFFER_TIME = 120; // 2 minutes

    public function updateRefreshToken(string $tmpToken): void
    {
        $initValues       = ['refresh_token' => '', 'expires_in' => ''];
        $refreshTokenData = ProxyApi::getInstance()->getRefreshToken($tmpToken) + $initValues;

        if (!empty($refreshTokenData['expires_in'])) {
            $refreshTokenData['expires_in'] = $refreshTokenData['expires_in'] + time();
        }

        $this->updateConfig($refreshTokenData, \XLite\Core\Auth::getInstance()->getProfile());
    }

    /**
     * Can be called from A zone only
     */
    public function resetRefreshToken(): void
    {
        $this->updateConfig(['refresh_token' => '', 'expires_in' => '', 'access_token' => ''], \XLite\Core\Auth::getInstance()->getProfile());
    }

    /**
     * Can be called from A and C zone
     */
    public function getAccessToken(array $configuration, $forceRenewAccessToken = false): ?string
    {
        $configRepo = Database::getRepo(Config::class);

        /** @var Config $accessTokenConf */
        $accessTokenConf = $this->getConfigByName($configRepo, $configuration, 'access_token');
        if (!$forceRenewAccessToken && $accessTokenConf && $accessTokenConf->getValue()) {
            [$expired, $accessToken] = explode('::', $accessTokenConf->getValue(), 2);

            if ($expired >= (time() + static::BUFFER_TIME)) {
                return $accessToken;
            }
        }

        /** @var Config|null $refreshTokenConf */
        $refreshTokenConf = $this->getConfigByName($configRepo, $configuration, 'refresh_token');
        /** @var Config|null $refreshTokenExpiresInConf */
        $refreshTokenExpiresInConf = $this->getConfigByName($configRepo, $configuration, 'expires_in');

        if (
            !$refreshTokenConf
            || empty($refreshTokenConf->getValue())
            || $refreshTokenExpiresInConf->getValue() <= time()
        ) {
            return null;
        }

        $accessTokenData = ProxyApi::getInstance()->getAccessToken($refreshTokenConf->getValue(), $configuration);
        $this->saveAccessTokenData($accessTokenData, $configuration);

        return $accessTokenData['access_token'] ?? null;
    }

    public function getConnectUrl(): string
    {
        return ProxyApi::getInstance()->getUPSLoginUrl();
    }

    public function getResetUrl(): string
    {
        $params = [];

        return \XLite::getInstance()->getShopURL(
            \XLite\Core\Converter::buildURL('ups_login', 'reset', $params, \XLite::getAdminScript())
        );
    }

    public function hasRefreshToken(): bool
    {
        return (bool)\XLite\Core\Config::getInstance()->XC->UPS->refresh_token;
    }

    protected function getConfigByName($configRepo, array $configuration, string $name)
    {
        return $configRepo->findOneBy([
            'category' => 'XC\UPS',
            'name'     => $name,
        ]);
    }

    protected function updateConfig(array $data, ?Profile $vendor = null): void
    {
        $configRepo     = Database::getRepo(Config::class);
        $changedOptions = [];
        $upsData        = $configRepo->findBy(['category' => 'XC\UPS', 'name' => array_keys($data)]);
        /** @var \XLite\Model\Config $upsDataItem */

        foreach ($upsData as $upsDataItem) {
            if (
                isset($data[$upsDataItem->getName()])
                && $upsDataItem->getValue() !== $data[$upsDataItem->getName()]
            ) {
                $upsDataItem->setValue($data[$upsDataItem->getName()]);
                $changedOptions[] = $upsDataItem;
            }
        }

        if ($changedOptions) {
            Database::getEM()->flush();
            \XLite\Core\Config::updateInstance();
        }
    }

    protected function saveAccessTokenData(?array $accessTokenData, array $configuration): void
    {
        if ($accessTokenData) {
            $data = [
                'access_token'  => $accessTokenData['access_token_expires_in'] + time() . '::' . $accessTokenData['access_token'],
                'refresh_token' => $accessTokenData['refresh_token'],
                'expires_in'    => $accessTokenData['refresh_token_expires_in'] + time(),
            ];
            $this->updateConfig($data);
        }
    }
}
