<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace XC\UPS\Core;

use JsonException;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\Exception\HttpExceptionInterface;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use XC\UPS\Core\Exception\ProxyApiException;
use XLite\Base\Singleton;
use XLite\InjectLoggerTrait;

class ProxyApi extends Singleton
{
    use InjectLoggerTrait;

    public const PROXY_HOST = 'https://ups-xc-proxy.xcart.com';

    public const LOGIN_ENDPOINT = '/login-to-ups';

    public const ACCESS_TOKEN_ENDPOINT = '/oauth/access-token';

    public const REFRESH_TOKEN_ENDPOINT = '/oauth/refresh-token';

    public function getUPSLoginUrl(): string
    {
        $params = $this->getUPSLoginUrlRequestParams();

        return static::PROXY_HOST . static::LOGIN_ENDPOINT . '?' . http_build_query($params);
    }

    public function getRefreshToken(string $tempToken): array
    {
        $this->getLogger('XC-UPS')->debug('Call getRefreshToken proxy method');
        $params = ['temp_token' => $tempToken];

        return $this->request(
            'POST',
            static::PROXY_HOST . static::REFRESH_TOKEN_ENDPOINT,
            $params
        );
    }

    public function getAccessToken(string $refreshToken, array $configuration): ?array
    {
        $this->getLogger('XC-UPS')->debug('Call getAccessToken proxy method');
        $params = $this->getAccessTokenRequestParams($refreshToken, $configuration);

        return $this->request(
            'POST',
            static::PROXY_HOST . static::ACCESS_TOKEN_ENDPOINT,
            $params
        );
    }

    protected function getAccessTokenRequestParams(string $refreshToken, array $configuration): array
    {
        return [
            'refresh_token' => $refreshToken,
            'xcart_domain'  => $this->getXcartDomain(),
        ];
    }

    protected function getUPSLoginUrlRequestParams(): array
    {
        return ['xcart_url' => $this->getXcartReturnUrl()];
    }

    /**
     * @throws HttpExceptionInterface
     * @throws JsonException
     * @throws TransportExceptionInterface
     */

    protected function request(string $method, string $endpoint, array $postData = []): array
    {
        $response = HttpClient::create()->request($method, $endpoint, $postData ? ['json' => $postData] : []);

        if ($response->getStatusCode() === 200) {
            return json_decode(
                $response->getContent(),
                true,
                512,
                JSON_THROW_ON_ERROR
            );
        }

        $this->getLogger('XC-UPS')->debug('Fail response: ' . $response->getContent());
        throw new ProxyApiException($response->getContent());
    }

    protected function getXcartReturnUrl(): string
    {
        return \XLite::getInstance()->getShopURL(
            \XLite\Core\Converter::buildURL('ups_login', 'return')
        );
    }

    protected function getXcartDomain(): string
    {
        return parse_url(\XLite::getInstance()->getShopURL(), PHP_URL_HOST);
    }
}
