/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * panel.js
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

define('themetweaker/panel_before', ['js/jquery'], function () {
  var isExpanded = jQuery.cookie('ThemeTweaker_isExpanded') !== 'false';
  if (isExpanded) {
    var sidebarGutter = 15;
    var sidebarWidth = jQuery('#themetweaker-panel .themetweaker-sidebar').outerWidth();
    var newMainWidth = (window.innerWidth - sidebarWidth) <= 1200
      ? window.innerWidth - sidebarWidth - 2 * sidebarGutter
      : '';

    _.each(jQuery('.container'), function (container) {
      var elem = jQuery(container);
      elem.css('width', newMainWidth);
    });

    var mainElem = jQuery('#main');
    mainElem.css('width', newMainWidth);
  }
});

define('themetweaker/panel_start', ['js/vue/vue', 'ready'], function (XLiteVue) {
  if (xcart.isDeveloperMode) {
    Vue.config.devtools = true;
  }

  XLiteVue.start('#themetweaker-panel-loader-point');
});

define('themetweaker/panel', ['js/vue/vue', 'vue/vuex', 'vue/vue.loadable', 'themetweaker/store', 'themetweaker/getters', 'themetweaker/modals', 'vue/eventbus'], function (XLiteVue, Vuex, Loadable, Store, Getters, ModalsMixin, EventBus) {
  XLiteVue.component('xlite-themetweaker-panel', {
    props: ['initialMode'],
    mixins: [ModalsMixin],
    store: Store,

    data: function () {
      return {
        mode: this.initialMode,
        isExpanded: jQuery.cookie('ThemeTweaker_isExpanded') !== 'false',
        reloading: false,
      }
    },

    created: function () {
      var self = this;

      EventBus.$on('blockPanel', function () {
        self.reloading = true;
      });

      EventBus.$on('addBannerToRemove', function(arg) {
        self.addBannerToRemove(arg);
      });

      EventBus.$on('unblockPanel', function () {
        self.reloading = false;
      });

      EventBus.$on('triggerSave', function () {
        EventBus.$once('completed.save', function (callback) {
          if (_.isFunction(callback)) {
            callback();
          } else {
            EventBus.$emit('unblockPanel');
          }
        });
        EventBus.$once('failed.save', function () {
          EventBus.$emit('unblockPanel');
        });
        self.savePanel();
      });

      EventBus.$on('triggerHide', function () {
        self.hidePanel();
      });

      EventBus.$on('triggerSwitchTab', function (mode) {
        self.switchTab(mode);
      });
    },

    beforeMount: function () {
      xcart.trigger('themetweaker-panel.activate', this);
    },

    mounted: function () {
      document.body.classList.add('storefront-builder-active');
      $('[data-toggle="tooltip"]', this.$el).tooltip();
      xcart.trigger('themetweaker-panel.ready', this);
      this.$panel = jQuery('.themetweaker-panel', this.$el);
      this.$sidebar = jQuery('.themetweaker-sidebar', this.$el);
      jQuery(this.$el).removeClass('themetweaker-panel--initial');
      this.initializeSidebarSwitcher();
      this.initializePanelTooltips();

      var self = this;
      this.$nextTick(function () {
        if (self.BOTTOM_PANEL_MODES.includes(this.mode)) {
          self.initializePanelHeight();
          if (self.isExpanded === true) {
            self.autoResizeFooter();
          }
          self.autoResizeFooter();
          self.enableResizing();
        }

        if (self.isExpanded === true) {
          self.$panel.removeClass('expand-transition');
          self.autoResizeContent();
        }

        self.autoResizeSidebar();
        self.enablePanelInteractionWhileInPopup();
      });

      jQuery(window).resize(_.bind(this.autoResizeSidebar, this));

      window.addEventListener('stickyHeaderElements', function() {
        const themeTweakerHeader = document.querySelector('.themetweaker-panel--header');
        if (themeTweakerHeader) {
          window.stickyHeaderHeight += themeTweakerHeader.getBoundingClientRect().height;
        }
      });

      window.addEventListener(
        'updateStickyElementsForPopup',
        function() {
          const themeTweakerPanel = document.querySelector('.themetweaker-sidebar');
          if (themeTweakerPanel) {
            window.stickyElementsForPopup.sizes.left += themeTweakerPanel.getBoundingClientRect().width;
          }
          const themeTweakerHeader = document.querySelector('.themetweaker-panel--header');
          if (themeTweakerHeader) {
            window.stickyElementsForPopup.top = themeTweakerHeader;
            window.stickyElementsForPopup.sizes.top += themeTweakerHeader.getBoundingClientRect().height;
          }
        }
      );
    },

    computed: _.extend({},
      Vuex.mapGetters([
        'switcher',
        'saveAvailable',
        'isChanged',
        'bannersRemoved'
      ]),

      Vuex.mapGetters('webmaster', [
        'hasWebmasterRevertedTemplates'
      ]), {

        MIN_HEIGHT: function () {
          return 100;
        },

        MAX_HEIGHT: function () {
          return 500;
        },

        PANEL_INITIAL_HEIGHT: function() {
          return 210;
        },

        PANEL_ENABLED_HEIGHT: function () {
          return 'ThemeTweaker.panelHeight.enabled';
        },

        PANEL_DISABLED_HEIGHT: function () {
          return this.MIN_HEIGHT;
        },

        BOTTOM_PANEL_MODES: function () {
          return [
            'webmaster',
            'custom_css'
          ];
        },

        SIDEBAR_GUTTER: function () {
          return 15;
        },

        panelClasses: function () {
          return {
            'reloading': this.reloading,
            'reloading-animated': this.reloading,
            'expanded': this.isExpanded,
            'collapsed': !this.isExpanded
          }
        },

        shouldShowSaveConfirm: function () {
          return this.mode === 'webmaster' && this.hasWebmasterRevertedTemplates;
        }
      }),

    watch: {
      isExpanded: function (value) {
        jQuery.cookie('ThemeTweaker_isExpanded', value);

        var switcher = jQuery('.sidebar-switcher', this.$panel);
        switcher.removeClass('fa-angle-left fa-angle-right')
          .addClass(value ? 'fa-angle-left' : 'fa-angle-right');
        switcher.attr('title', value ? xcart.t('Collapse') : xcart.t('Expand'))
          .tooltip('fixTitle');
      }
    },

    transitions: {
      'expand': {
        beforeEnter: function (el) {
          this.$panel.addClass('expand-transition');
        },
        afterEnter: function (el) {
          this.$panel.removeClass('expand-transition');
          this.autoResizeContent();
          this.autoResizeFooter();
        },
        beforeLeave: function (el) {
          this.$panel.addClass('expand-transition');
        },
        afterLeave: function (el) {
          this.autoResizeFooter();
        },
      }
    },

    methods: {
      ...Vuex.mapActions([
        'toggleSwitcher',
        'addBannerToRemove'
      ]),

      initializePanelHeight: function () {
        var savedHeight = this.getPanelHeight();

        if (savedHeight) {
          this.$panel.css('height', savedHeight);
        }
      },

      initializeSidebarSwitcher: function () {
        var switcher = jQuery('<span class="fa sidebar-switcher"></span>')
          .addClass(this.isExpanded ? 'fa-angle-left' : 'fa-angle-right');
        switcher.tooltip({
          'title': this.isExpanded ? xcart.t('Collapse') : xcart.t('Expand'),
          'html': true,
          'placement': 'bottom'
        });
        this.$sidebar.parent().prepend(switcher);
        switcher.click(_.bind(function () {
          if (this.isExpanded) {
            this.hidePanel();
          } else {
            this.showPanel();
          }
        }, this));
      },

      initializePanelTooltips: function () {
        jQuery('.themetweaker-panel-tooltip').tooltip({
          trigger: 'manual'
        }).hover(function () {
          var elem = $(this);
          var hoverTimeout = setTimeout(function () {
            elem.tooltip('show');
            elem.siblings('.tooltip').hover(function () {
              $(this).addClass('tooltip-hovered');
            }, function () {
              $(this).removeClass('tooltip-hovered');
              elem.tooltip('hide');
            });
          }, 250);
          elem.data('hoverTimeout', hoverTimeout);
        }, function () {
          var elem = $(this);
          clearTimeout(elem.data('hoverTimeout'));
          setTimeout(function () {
            if (elem.siblings('.tooltip.tooltip-hovered').length === 0) {
              elem.tooltip('hide');
            }
          }, 150);
        });
      },

      enablePanelInteractionWhileInPopup: function () {
        jQuery.widget("ui.dialog", jQuery.ui.dialog, {
          _allowInteraction: function (event) {
            return !!$(event.target).closest(".themetweaker-panel").length || this._super(event);
          }
        });
      },

      enableResizing: function () {
        this.$panel.resizable({
          handles: "n",
          maxHeight: this.MAX_HEIGHT,
          minHeight: this.MIN_HEIGHT,
        });

        this.$panel.on("resize", _.bind(this.autoResizeContent, this));
        this.$panel.on("resizestop", _.bind(this.autoResizeContent, this));
        this.$panel.on("resizestop", _.bind(this.autoResizeFooter, this));
      },

      autoResizeContent: function () {
        var panelHeight = this.$panel.height();
        var tabs = this.$panel.find('[data-panel-tabs]');
        jQuery('[data-panel-content]', this.$panel).css('height', panelHeight - tabs.height());

        this.setPanelHeight(panelHeight);
        EventBus.$emit('panel.resize');
      },

      autoResizeFooter: function () {
        var height = this.$panel.height();
        if (this.$panel.hasClass('themetweaker-panel-empty'))
          height = 0;
        jQuery('#footer-area').css('margin-bottom', height);
        this.$panel.attr('data-height', height);
      },

      autoResizeSidebar: function () {
        var sidebarWidth = this.$sidebar.outerWidth();
        var newMainWidth = (window.innerWidth - sidebarWidth) <= 1200
          ? window.innerWidth - sidebarWidth - 2 * this.SIDEBAR_GUTTER
          : '';
        var leftMargin = sidebarWidth;
        var rightMargin = 0;
        if (!this.isExpanded) {
          newMainWidth = '';
          rightMargin = 0;
        }

        const pageWrapper = jQuery('#page-wrapper');
        if (pageWrapper[0]) {
          let maxPageWrapperWidth = parseInt(getComputedStyle(pageWrapper[0]).maxWidth);
          if (!maxPageWrapperWidth)
            maxPageWrapperWidth = 1440;
          const marginFix = window.innerWidth - maxPageWrapperWidth - leftMargin - rightMargin;
          if (marginFix > 0) {
            leftMargin += Math.ceil(marginFix / 2);
            rightMargin += Math.floor(marginFix / 2);
          }
        }

        var mainElem = jQuery('#main');
        mainElem.css('width', newMainWidth);
        jQuery('.desktop-header').css({
          'padding-left': sidebarWidth
        });
        jQuery('.sf-toolbarreset').css({
          'padding-left': sidebarWidth
        });
        _.each(jQuery('.container'), function (container) {
          var elem = jQuery(container);
          elem.css('width', newMainWidth);
        });

        if (window.innerWidth < 768) {
          this.isExpanded = false;
        }

        const body = jQuery('body');
        pageWrapper.css('margin-left', leftMargin);
        pageWrapper.css('margin-right', rightMargin);
        jQuery('.roulette-wrapper').css('left', leftMargin);
        body.addClass('sidebar-autoresized');
        body.toggleClass('sidebar-expanded', this.isExpanded);
        popup.reposition();
      },

      sendSwitchRequest: function (mode) {
        var data = {};
        data[xliteConfig.form_id_name] = xliteConfig.form_id;

        if (mode) {
          data['mode'] = mode;
        }

        return xcart.post(
          {
            base: xliteConfig.admin_script,
            target: 'theme_tweaker',
            action: 'switch_mode'
          },
          null,
          data
        );
      },

      sendCloseRequest: function () {
        var data = {};
        data[xliteConfig.form_id_name] = xliteConfig.form_id;
        return xcart.post(
          {
            base: xliteConfig.admin_script,
            target: 'theme_tweaker',
            action: 'disable'
          },
          null,
          data
        );
      },

      doSave: function() {
        var performSave = function () {
          if (this.saveAvailable && this.isChanged) {
            EventBus.$emit('blockPanel');
            EventBus.$emit('action.save');
          }
        };

        if (this.shouldShowSaveConfirm) {
          this.saveConfirm(performSave, null);
        } else {
          performSave.apply(this);
        }
      },

      savePanel: function () {
        const self = this;
        if (this.bannersRemoved.length > 0) {
          EventBus.$emit('showDeleteBannerConfirm', function() {
            self.doSave();
          }, function() {}, 'banner');
        } else {
          self.doSave();
        }
      },

      showPanel: function () {
        this.isExpanded = true;

        var self = this;
        this.$nextTick(function () {
          self.autoResizeSidebar();
          document.body.fireEvent('themetweaker_panel_shown', 'CustomEvent');
        });
      },

      hidePanel: function () {
        this.isExpanded = false;

        var self = this;
        this.$nextTick(function () {
          self.autoResizeSidebar();
          document.body.fireEvent('themetweaker_panel_hidden', 'CustomEvent');
        });
      },

      switchTab: function (mode) {
        var completion = function (callback) {
          // TODO: ignores callback right now because only webmaster mode is using it to reload the page, needs to be reimplemented
          EventBus.$emit('panel.switch');
          this.sendSwitchRequest(mode);
        };

        var onSave = function () {
          EventBus.$once('completed.save', _.bind(completion, this));
          EventBus.$once('failed.save', function () {
            EventBus.$emit('unblockPanel');
          });
          this.savePanel();
        };

        var onDiscard = function () {
          EventBus.$emit('blockPanel');
          completion.apply(this);
        };

        if (this.mode && this.isChanged) {
          this.exitConfirm(onSave, onDiscard);
        } else {
          onDiscard.apply(this);
        }
      },

      getPanelHeight: function () {
        if (this.BOTTOM_PANEL_MODES.includes(this.mode)) {
          if (!sessionStorage.getItem(this.PANEL_ENABLED_HEIGHT)) {
            this.setPanelHeight();
          }

          var value = sessionStorage.getItem(this.PANEL_ENABLED_HEIGHT);

          return value !== null
            ? Math.min(Math.max(this.MIN_HEIGHT, value), this.MAX_HEIGHT)
            : null;
        } else {
          return this.PANEL_DISABLED_HEIGHT;
        }
      },

      setPanelHeight: function (value = 0) {
        if (value === 0) {
          value = this.PANEL_INITIAL_HEIGHT;
        }
        var height = Math.min(Math.max(this.MIN_HEIGHT, value), this.MAX_HEIGHT);
        if (this.BOTTOM_PANEL_MODES.includes(this.mode)) {
          sessionStorage.setItem(this.PANEL_ENABLED_HEIGHT, height);
        }
      },

      goAdminPanel: function () {
        window.location.href = URLHandler.buildURL(
          {
            'base': xliteConfig.admin_script,
            'target': 'theme_tweaker_switcher',
            'switch': 'off'
          }
        );
      },
    }
  });
});
