/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Layout options component
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

define('themetweaker/layout_options', ['js/vue/vue', 'vue/vuex', 'vue/eventbus', 'themetweaker/blocks_list'], function (XLiteVue, Vuex, EventBus, BlocksListMixin) {
  XLiteVue.component('xlite-layout-options', {
    props: ['preset', 'initialResetAvailable', 'isCustomCssEnabled'],
    mixins: [BlocksListMixin],

    data: function () {
      return {
        headerGroupItems: this.getHeaderGroupItems(),
        footerGroupItems: this.getFooterGroupItems(),
        settingsGroupItems: this.getSettingsGroupItems()
      };
    },

    created: function () {
      EventBus.$on('layout-type.selected', _.bind(function (value, sender) {
        EventBus.$emit('blockPanel');
        var params = {
          'group': sender.group,
          'type': value,
          'returnURL': window.location.href
        };

        params[xliteConfig.form_id_name] = xliteConfig.form_id;

        xcart.post(
          {
            base: xliteConfig.admin_script,
            target: 'layout_edit',
            action: 'switch_layout_type'
          },
          null,
          params
          )
          .done(_.bind(this.onSwitchSuccess, this))
          .fail(_.bind(this.onSwitchFail, this));
      }, this));

      EventBus.$on('action.reset', _.bind(function () {
        EventBus.$emit('blockPanel');
        var params = {
          'preset': this.preset,
          'returnURL': window.location.href
        };

        params[xliteConfig.form_id_name] = xliteConfig.form_id;

        xcart.post(
          {
            base: xliteConfig.admin_script,
            target: 'layout_edit',
            action: 'reset_layout'
          },
          null,
          params
          )
          .done(_.bind(this.onResetSuccess, this))
          .fail(_.bind(this.onResetFail, this));
      }, this));

      EventBus.$on('action.remove-entity', _.bind(function (removeData) {
        if (removeData && removeData.id && removeData.id.indexOf('banner_slide_image_') === 0) {
          removeData.element.css('opacity', '0.4');
          jQuery('[data-removeid="' + removeData.id + '"]').css('opacity', '0.4');
          var collapsible_element = removeData.element.children('.collapse').eq(0);
          let updateData = {delete_banner: removeData.id};
          if (collapsible_element.attr('aria-expanded')) {
            collapsible_element.collapse('hide');
          }
          this.updateImage({key: removeData.id, value: updateData});

          EventBus.$emit('addBannerToRemove', { id: removeData.id });
        }
      }, this));

      EventBus.$on('action.add-entity', _.bind(function (addData) {
        if (addData && addData.type) {
          let updateData = {add_banner: this.preset + '.' + addData.type + '.' + addData.subtype};
          this.updateImage({key: 'banner_slide_image_' + _.uniqueId('#'), value: updateData});
        }
      }, this));

      EventBus.$on('form-model-prop-updated', _.bind(function (path, value, sender) {
        var updateData = {};
        if (typeof (sender.temp_id) !== 'undefined' && sender.temp_id !== null) {
          updateData.temp_id = sender.temp_id;

          if (typeof (sender.alt) !== 'undefined') {
            updateData.alt = sender.alt;
          }
        }

        if (typeof (sender.del) !== 'undefined') {
          updateData.is_delete = sender.del;
        }

        if (typeof (sender.alt) !== 'undefined'
          && sender.alt !== sender.initialAlt
        ) {
          updateData.alt = sender.alt;
        }

        if (_.isEqual(updateData, {is_delete: false}) || _.isEmpty(updateData)) {
          updateData = null;
        }

        this.updateImage({key: sender.basePath, value: updateData});
      }, this));
    },

    mounted: function () {
      this.setResetAvailable(this.initialResetAvailable);
      this.initSettingsState();
    },

    computed: _.extend({},
      Vuex.mapState('layoutEditor', [
        'changeset',
        'resetAvailable',
        'settingsState'
      ]), {}),

    methods: _.extend({},
      Vuex.mapActions('layoutEditor', [
        'updateImage',
        'updateSettingsState',
        'updateOriginalSettingsState',
        'setResetAvailable'
      ]), {

        onSwitchSuccess: function (event) {
          xcart.trigger('message', {type: 'info', message: xcart.t('Layout type was successfully changed')});
        },

        switchTab: function (mode) {
          EventBus.$emit('triggerSwitchTab', mode);
        },

        onSwitchFail: function (event) {
          xcart.trigger('message', {type: 'error', message: xcart.t('Unable to change the layout type')});
          EventBus.$emit('unblockPanel');
        },

        onResetSuccess: function (event) {
          xcart.trigger('message', {type: 'info', message: xcart.t('Layout was successfully reset')});
        },

        onResetFail: function (event) {
          xcart.trigger('message', {type: 'error', message: xcart.t('Unable to reset layout')});
          EventBus.$emit('unblockPanel');
        },

        onCustomCSSSwitchChange: function (event) {
          this.updateSettingsState({key: 'customCss', value: event.target.checked ? '1' : '0'});
        },

        initSettingsState: function () {
          this.updateOriginalSettingsState({key: 'customCss', value: this.isCustomCssEnabled ? '1' : '0'});
          this.updateSettingsState({key: 'customCss', value: this.isCustomCssEnabled ? '1' : '0'});
        },

        getHeaderGroupItems: function () {
          var items = [];

          if (jQuery('.navbar').length > 0) {
            items.push({
              id: 'navbar',
              name: xcart.t('Primary Menu'),
              weight: 200,
              settings_link: {'target': 'pages', 'page': 'menus_P'},
              element: jQuery('.navbar')
            });
          }

          return this.prepareListItems(items);
        },

        getFooterGroupItems: function () {
          var items = [];

          items.push({
            id: 'footer-subscription',
            name: xcart.t('Sign up for news'),
            weight: 100,
            help: xcart.t('You will be able to manage this block in the next version of Storefront Builder'),
            element: jQuery('.subscription-form-block')
          });

          items.push({
            id: 'bottom-menu',
            name: xcart.t('Footer Menu'),
            weight: 200,
            settings_link: {'target': 'pages', 'page': 'menus_F'},
            element: jQuery('#footer-area .footer-menu')
          });

          return this.prepareListItems(items);
        },

        getSettingsGroupItems: function () {
          var items = [];

          items.push({
            id: 'custom-css',
            name: xcart.t('Custom CSS'),
            weight: 100,
            help: xcart.t('This tool allows you to add your Custom CSS. Read more about using Custom CSS.'),
            goToTab: 'custom_css',
            switcherState: this.isCustomCssEnabled,
            switcherHandler: _.bind(this.onCustomCSSSwitchChange, this)
          });

          items.push({
            id: 'template-editor',
            name: xcart.t('Template Editor'),
            weight: 300,
            help: xcart.t('This tool allows you to edit Templates. Read more about using Template Editor.'),
            goToTab: 'webmaster'
          });

          items.push({
            id: 'labels-editor',
            name: xcart.t('Labels Editor'),
            weight: 200,
            help: xcart.t('This tool allows you to edit Labels. Read more about using Labels Editor.'),
            goToTab: 'labels_editor'
          });

          items.push({
            id: 'seo',
            name: xcart.t('SEO'),
            weight: 400,
            settings_link: {'target': 'settings', 'page': 'CleanURL'}
          });

          return this.prepareListItems(items);
        }
      })
  });
});
