/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Layout groups component
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

define('themetweaker/layout_groups', ['js/vue/vue', 'vue/vuex', 'vue/eventbus', 'themetweaker/blocks_list'], function (XLiteVue, Vuex, EventBus, BlocksListMixin) {
  XLiteVue.component('xlite-layout-groups', {
    props: ['preset', 'groupsPreset'],
    mixins: [BlocksListMixin],

    created: function () {
    },

    mounted: function () {
      this.assignHandlers();
      this.updateHiddenItemsLinks();
      this.handleHeaderSectionClicks();
    },

    data: function () {
      return (
        (xcart.getTarget() === 'checkout')
        ? {}
        : { layoutGroups: this.getLayoutGroups() }
      );
    },

    computed: {
      WEIGHT_STEP: function () {
        return 10;
      }
    },

    methods: _.extend({},
      Vuex.mapActions('layoutEditor', [
        'updateChangeset',
        'updateImage'
      ]), {

        updateHeaderSection: function() {
          const hiddenChildrenCount = this.$headerHiddenSection.children().length;
          this.$link.toggleClass('hidden', hiddenChildrenCount === 0);
          this.$link.html(hiddenChildrenCount.toString().concat(' ').concat(xcart.t('block(s)')));
        },

        handleHeaderSectionClicks: function() {
          this.$link = jQuery('.hidden-items-link--header');
          this.$headerHiddenSection = jQuery('.group-items.hidden-group[data-list="layout.top.wide"]');
          this.$headerSections = jQuery('[data-list="layout.top.wide"]');
          this.$headerSections.find('.themetweaker-panel-hide-switcher').on('click', this.updateHeaderSection);
          const self = this;
          this.$link.bind('click', function() {
            self.$headerHiddenSection.toggleClass('hidden');
          });
          this.updateHeaderSection();
        },

        assignHandlers: function () {
          xcart.bind('layout.listElemMoved', _.bind(this.onListElemMoved, this));
        },

        updateHiddenItemsLinks: function () {
          let hiddenGroups = jQuery('.group-items.hidden-group', this.$el);
          hiddenGroups.each(function (index, elem) {
            let count = jQuery(elem).find('.group-item').length;
            let link = jQuery(elem).closest('.options-layout-group').find('.hidden-items-link');
            link.text(count + ' ' + xcart.t('block(s)')).removeClass('hidden');
            if (count === 0) {
              link.addClass('hidden');
            }
          });
        },

        hiddenItemsLinkHandler: function (event) {
          jQuery(event.target).closest('.options-layout-group').find('.group-items.hidden-group').toggleClass('hidden');
        },

        addChange: function (id, list, weight, mode) {
          this.updateChangeset({
            id: id,
            list: list,
            weight: weight,
            mode: mode
          });
        },

        onListElemMoved: function (event, data) {
          let dragged = jQuery(data.dragged);
          let related = jQuery(data.related);
          let draggedId = dragged.data('id');
          let relatedId = related.data('id');
          let fromList = jQuery(data.from).data('list');
          let toList = jQuery(data.to).data('list');
          let fromGroup = jQuery('.list-items-group[data-list^="' + fromList + '"]').closest('[data-group]').data('group');
          let toGroup = jQuery('.list-items-group[data-list^="' + toList + '"]').closest('[data-group]').data('group');

          let draggedItem,
            relatedItem;

          this.layoutGroups.forEach(function (group) {
            group.items.forEach(function (item) {
              if (item.id === draggedId) {
                draggedItem = item;
              }

              if (item.id === relatedId) {
                relatedItem = item;
              }
            })
          });

          if (draggedItem) {
            if (relatedItem) {
              if (data.willInsertAfter) {
                draggedItem.element.insertAfter(relatedItem.element);
              } else {
                draggedItem.element.insertBefore(relatedItem.element);
              }

            } else {
              let groupElem = jQuery('.list-items-group[data-list="' + toList + '"]').eq(-1);
              if (groupElem.length > 0) {
                draggedItem.element.appendTo(groupElem);
              }
            }

            this.calculateWeight(draggedItem.element);

            if (fromGroup !== toGroup) {
              draggedItem.element.data('display', toGroup);
              xcart.trigger(
                'layout.block.reload',
                {
                  id: draggedId,
                  displayGroup: toGroup,
                }
              );
            }

            // Adjust target list
            const draggedElementToList = jQuery(draggedItem.element).closest('.list-items-group').data('list');
            if (draggedElementToList) {
              toList = draggedElementToList;
            }

            // Update banner location in settings
            if (
              jQuery(draggedItem.element).data('removeid')
              && jQuery(draggedItem.element).data('removeid').indexOf('banner_slide_image_') === 0
            ) {
              jQuery(draggedItem.element).find('.cycle-slideshow').cycle('destroy');

              this.updateImage({
                key: jQuery (draggedItem.element).data('removeid'),
                value: {change_banner_type: toList}
              });

              jQuery(draggedItem.element).find('.cycle-slideshow').cycle();
            }

            this.addChange(draggedId, toList, draggedItem.element.data('weight'), 1);
          }
        },

        calculateWeight: function (element) {
          let prev = parseInt($(element).prev().data('weight')) || 0;
          let next = parseInt($(element).next().data('weight')) || 0;
          let weight;

          if (next) {
            weight = Math.ceil((next + prev) / 2);
          } else {
            weight = prev + this.WEIGHT_STEP
          }

          $(element).data('weight', weight);

          this.recalculateWeightsRecursive(element);

          return weight;
        },

        recalculateWeightsRecursive: function (element) {
          const weight = $(element).data('weight');
          const $next = $(element).next();

          if ($next.length && weight >= $next.data('weight')) {
            const newPosition = weight + this.WEIGHT_STEP;

            $next.data('weight', newPosition);

            this.addChange($next.data('id'), $next.data('list'), newPosition, 1);

            this.recalculateWeightsRecursive($next);
          }
        },

        getGroupIconClass: function (group) {
          let iconClass = this.groupsPreset;

          switch (group.list) {
            case 'sidebar.first':
              iconClass += '-left';
              break;
            case 'sidebar.second':
              iconClass += '-right';
              break;
            case 'layout.bottom.wide':
              iconClass = 'footer-group';
              break;
            default:
              iconClass += '-central';
          }

          return iconClass + '-icon';
        },

        // visibility modes: 0b01 (only hidden), 0b10 (only visible), 0b11 (all)
        getGroupItems: function (group, visibility) {
          return group.items.filter(function (item) {
            return visibility & 1 && item.is_hidden
              || visibility & 2 && !item.is_hidden
          });
        },

        getLayoutGroups: function () {
          switch (this.groupsPreset) {
            case 'left':
              return [
                this.getHeaderGroup(),
                this.getSidebarFirstGroup(),
                this.getCentralGroup(),
                this.getFooterGroup()
              ];
            case 'right':
              return [
                this.getHeaderGroup(),
                this.getCentralGroup(),
                this.getSidebarSecondGroup(),
                this.getFooterGroup()
              ];
            case 'three':
              return [
                this.getHeaderGroup(),
                this.getSidebarFirstGroup(),
                this.getCentralGroup(),
                this.getSidebarSecondGroup(),
                this.getFooterGroup()
              ];
            default:
              return [
                this.getHeaderGroup(),
                this.getCentralGroup(),
                this.getFooterGroup()
              ];
          }
        },

        getSidebarFirstGroup: function () {
          let self = this;
          let items = jQuery('#sidebar-first .list-item[data-list^="sidebar."]').toArray().reduce(function (blocks, elem) {
            let itemData = self.getItemData(jQuery(elem));
            if (!_.isEmpty(itemData)) {
              blocks.push(itemData);
            }

            return blocks;
          }, []);

          if (jQuery('.bannerBox.MainColumn-box.add-button-body').length > 0) {
            items = this.addItemAddBlock(items,'.bannerBox.MainColumn-box.add-button-body', 'sidebar-first');
          }

          return {
            'header': xcart.t('Left Column'),
            'list': 'sidebar.first',
            'items': this.prepareListItems(items)
          }
        },

        getSidebarSecondGroup: function () {
          let self = this;
          let items = jQuery('#sidebar-second .list-item[data-list^="sidebar."]').toArray().reduce(function (blocks, elem) {
            let itemData = self.getItemData(jQuery(elem));
            if (!_.isEmpty(itemData)) {
              blocks.push(itemData);
            }

            return blocks;
          }, []);

          if (jQuery('.bannerBox.SecondaryColumn-box.add-button-body').length > 0) {
            items = this.addItemAddBlock(items,'.bannerBox.SecondaryColumn-box.add-button-body', 'sidebar-second');
          }

          return {
            'header': xcart.t('Right Column'),
            'list': 'sidebar.second',
            'items': this.prepareListItems(items)
          }
        },

        getHeaderGroup: function () {
          let self = this;
          let items = jQuery('.list-item[data-list="layout.top.wide"]').toArray().reduce(function (blocks, elem) {
            let itemData = self.getItemData(jQuery(elem));
            if (!_.isEmpty(itemData)) {
              blocks.push(itemData);
            }

            return blocks;
          }, []);

          if (jQuery('.bannerBox.WideTop-box.add-button-body').length > 0) {
            items = this.addItemAddBlock(items, '.bannerBox.WideTop-box.add-button-body', 'top');
          }

          return {
            'header': '',
            'list': 'layout.top.wide',
            'items': this.prepareListItems(items)
          }
        },

        getCentralGroup: function () {
          let self = this;
          let items = [];
          let bottomButtons = [];
          ['center.top', 'center', 'center.bottom'].forEach(function (list) {
            let listItems = jQuery('.list-item[data-list="' + list + '"]').toArray().reduce(function (blocks, elem) {
              let itemData = self.getItemData(jQuery(elem));
              if (!_.isEmpty(itemData)) {
                if (jQuery(elem).data('isbutton')) {
                  bottomButtons.push(itemData)
                } else {
                  blocks.push(itemData)
                }
              }

              return blocks;
            }, []);

            items = items.concat(self.prepareListItems(listItems))
          })
          // put all buttons in the bottom of section
          if (!_.isEmpty(bottomButtons)) {
            items = items.concat(self.prepareListItems(bottomButtons))
          }

          if (jQuery('.bannerBox.StandardTop-box.add-button-body').length > 0) {
            items = this.addItemAddBlock(items,'.bannerBox.StandardTop-box.add-button-body', 'center');
          }

          return {
            'header': xcart.t('Central Column'),
            'list': 'center',
            'items': items
          }
        },

        getFooterGroup: function () {
          let self = this;
          let items = jQuery('.list-item[data-list="layout.bottom.wide"]').toArray().reduce(function (blocks, elem) {
            let itemData = self.getItemData(jQuery(elem));
            if (!_.isEmpty(itemData)) {
              blocks.push(itemData);
            }

            return blocks;
          }, []);

          if (jQuery('.bannerBox.WideBottom-box.add-button-body').length > 0) {
            items = this.addItemAddBlock(items, '.bannerBox.WideBottom-box.add-button-body', 'bottom');
          }

          return {
            'header': xcart.t('Footer'),
            'list': 'layout.bottom.wide',
            'items': this.prepareListItems(items)
          }
        },

        hideSwitcherHandler: function (item, event) {
          event.stopPropagation();
          let switcher = jQuery(event.target);
          let groupItem = switcher.closest('.group-item');
          let hiddenGroup = switcher.closest('.options-layout-group').find('.group-items.hidden-group');
          let movableGroup = switcher.closest('.options-layout-group').find('.group-items.movable-group');

          groupItem.css('opacity', '0');
          if (item.element.hasClass('list-item__hidden')) {
            this.showBlock(item.element);
            switcher.removeClass('layout-item-hidden').addClass('layout-item-visible');

            let followingElem;
            movableGroup.find('.group-item').each(function (index, elem) {
              let itemId = jQuery(elem).data('id');
              let listElem = jQuery('.list-item[data-id="' + itemId + '"]');
              if (item.element.closest('[data-group]').data('group') === 'center') {
                let centerLists = ['center.top', 'center', 'center.bottom'];
                if (listElem.data('list') === item.element.data('list')) {
                  if (listElem.length > 0
                    && listElem.data('weight') > item.element.data('weight')
                  ) {
                    followingElem = jQuery(elem);
                    return false;
                  }
                } else if (centerLists.indexOf(listElem.data('list')) > centerLists.indexOf(item.element.data('list'))) {
                  followingElem = jQuery(elem);
                  return false;
                }

              } else {
                if (listElem.length > 0
                  && listElem.data('weight') > item.element.data('weight')
                ) {
                  followingElem = jQuery(elem);
                  return false;
                }
              }
            });

            if (followingElem) {
              groupItem.insertBefore(followingElem);
            } else {
              movableGroup.append(groupItem);
            }

          } else {
            this.hideBlock(item.element);
            hiddenGroup.append(groupItem);
            switcher.removeClass('layout-item-visible').addClass('layout-item-hidden');
          }

          setTimeout(function () {
            groupItem.css('opacity', '');
          }, 50);

          this.updateHiddenItemsLinks();
        },

        hideBlock: function (block) {
          let list = block.closest('.list-items-group');
          block.listItem('hide');
          this.addChange(block.data('id'), list.data('list'), block.data('weight'), 2);
        },

        unhideBlock: function (block) {
          block.blur();
          let list = block.closest('.list-items-group');
          this.addChange(block.data('id'), list.data('list'), block.data('weight'), 1);
        },

        showBlock: function (block) {
          if (!block.listItem('isHidden')) {
            return;
          }

          if (block.listItem('isSidebarHidden')) {
            this.showSidebarHiddenBlock(block);
          } else {
            block.listItem('show');
            this.unhideBlock(block);
            block.listItem('focus');
          }
        },

        getCenterListContainer: function () {
          return $('.list-container[data-group="center"]');
        },

        showSidebarHiddenBlock: function (block) {
          this.getCenterListContainer().each(function() {
            $(this).data('controller').move(block);
          });
          block.listItem('show');
          block.listItem('focus');
        },
      })
  });
});
