/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Labels editor component
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

define('themetweaker/labels_editor', ['js/vue/vue', 'vue/vuex', 'vue/eventbus'], function (XLiteVue, Vuex, EventBus) {
  XLiteVue.component('xlite-labels-editor', {
    created () {
      this.HIGHLIGHT_MODE_KEY = 'highlight-labels-mode';
    },

    mounted: function () {
      xcart.trigger('labels-editor.ready', this);

      this.hideSaveButton();
      this.initializePopovers();
      this.setSwitcherState(this.isHighlightModeEnabled());
      this.setupNewLabelsObserver();
    },

    watch: {
      'switcher': function(value, oldValue) {
        let highlightModeStatus = '';

        if (value) {
          xcart.trigger('editable-label.enable');
          highlightModeStatus = 'on';
        } else {
          xcart.trigger('editable-label.disable');
          highlightModeStatus = 'off';
        }

        this.setHighlightModeStatus(highlightModeStatus);
      }
    },

    computed: Vuex.mapGetters([
      'switcher'
    ]),

    methods: _.extend(
      Vuex.mapActions([
        'setSwitcherState',
        'hideSaveButton'
      ]),{
        isHighlightModeEnabled: function () {
          return sessionStorage.getItem(this.HIGHLIGHT_MODE_KEY) !== 'off';
        },

        setHighlightModeStatus: function (value) {
          sessionStorage.setItem(this.HIGHLIGHT_MODE_KEY, value);
        },

        initializePopovers: function () {
          $(document).on('click', EditableLabel.closePopovers);
          xcart.autoload(EditableLabel, '.xlite-translation-label');
        },

        switchTab: function (mode) {
          EventBus.$emit('triggerSwitchTab', mode);
        },

        onSwitchChange: function (event) {
          this.$store.dispatch('toggleSwitcher', event.target.checked);
        },

        // Track dynamically added/replaced labels and initialize them
        setupNewLabelsObserver: function () {
          const observer = new MutationObserver(function(mutations) {
            mutations.forEach(function(mutation) {
              mutation.addedNodes.forEach(function(node) {
                if (!node) {
                  return;
                }
                let el = node.querySelector('.xlite-translation-label');
                if (el) {
                  const $el = jQuery(el);
                  // Imitate xcart.autoload
                  !$el.data('controller') && $el.data('controller', new EditableLabel($el));
                }
              });
            });
          });
          observer.observe(document.querySelector("#main-wrapper"), { subtree: true, childList: true });
        },
      })
  });
});
