/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Inline editor mode component
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

define('themetweaker/inline_editor', ['js/vue/vue', 'vue/vuex', 'vue/eventbus'], function (XLiteVue, Vuex, EventBus) {
  XLiteVue.component('xlite-inline-editor', {

    created: function () {
      var self = this;

      EventBus.$on('action.save', function () {
        self.submitChanges();
      });
    },

    mounted: function() {
      var self = this;
      xcart.trigger('inline-editor.ready', this);
      this.$nextTick(function () {
        jQuery(self.$el).removeClass('inline-editor--initial');
      });
      this.assignHandlers();
      this.setSwitcherState(true);
      this.hideSwitcher();
      xcart.autoload(InlineEditableController);
    },

    watch: {
      'switcher': function (value, oldValue) {
        if (oldValue !== null) {
          this.toggleEditor(value);
        }
      }
    },

    computed: _.extend(
      Vuex.mapState('inlineEditor', [
        'changeset',
        'images',
        'videos',
      ]),

      Vuex.mapGetters([
        'switcher'
      ])
    ),

    methods: _.extend(
      Vuex.mapActions([
        'setSwitcherState',
        'hideSwitcher'
      ]),

      Vuex.mapActions('inlineEditor', [
        'clearChangeset',
        'updateChangeset',
        'clearImages',
        'clearVideos',
        'updateImages',
        'updateVideos'
      ]), {

        assignHandlers: function () {
          xcart.bind('inline_editor.image.inserted', _.bind(this.onImageInserted, this));
          xcart.bind('inline_editor.video.inserted', _.bind(this.onVideoInserted, this));
          xcart.bind('inline_editor.changed', _.bind(this.onChanged, this));
        },

        submitChanges: function () {
          var params = {
            changeset: this.changeset,
            videos: _.keys(this.videos),
            images: _.keys(this.images)
          };

          params[xliteConfig.form_id_name] = xliteConfig.form_id;

          xcart.post(
            {
              base: xliteConfig.admin_script,
              target: 'inline_editable',
              action: 'update_field'
            },
            null,
            params,
            {
              dataType: 'json',
            }
            )
            .done(_.bind(this.onSaveSuccess, this))
            .fail(_.bind(this.onSaveFail, this));
        },

        onChanged: function (event, data) {
          this.updateChangeset({key: data.fieldId, value: data.change});
        },

        onImageInserted: function (event, data) {
          this.updateImages({key: data.imageId, value: data.imageElement[0]});
        },

        onVideoInserted: function (event, data) {
          this.updateVideos({key: data.videoId, value: data.videoElement[0]});
        },

        onSaveSuccess: function (event, status, xhr) {
        },

        onSaveFail: function (event) {
          xcart.trigger('message', {type: 'error', message: xcart.t('Unable to save changes')});
          EventBus.$emit('failed.save');
        },

        toggleEditor: function (state) {
          if (state) {
            xcart.trigger('inline_editor.enable', this);
          } else {
            xcart.trigger('inline_editor.disable', this);
          }
        }
      })
  });
});
