/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Css hot editor panel
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

define('themetweaker/custom_css', ['js/vue/vue', 'vue/vuex', 'vue/eventbus'], function (XLiteVue, Vuex, EventBus) {
  XLiteVue.component('xlite-custom-css', {
    props: ['initial'],

    created: function () {
      var self = this;

      EventBus.$on('action.save', function () {
        var params = {
          use: self.use ? 1 : 0,
          code: self.content
        };

        params[xliteConfig.form_id_name] = xliteConfig.form_id;

        xcart.post(
          {
            base: xliteConfig.admin_script,
            target: 'custom_css',
            action: 'save',
          },
          undefined, params, { timeout: 45000, rpc: true }
          )
          .done(_.bind(self.onSaveSuccess, self))
          .fail(_.bind(self.onSaveFail, self));
      });

      EventBus.$on('panel.resize', function () {
        self.resizeTextarea();
      });
    },

    beforeMount: function () {
      xcart.trigger('custom-css.activate', this);
    },

    mounted: function () {
      xcart.trigger('custom-css.ready', this);
      this.setSwitcherState(this.initial);
      this.findElements();
      this.initTextarea();
      this.enableLiveReloading();
    },

    watch: {
      switcher: function(value, oldValue) {
        if (oldValue !== null) {
          var text = this.$css.text();

          if (value) {
            this.$css.replaceWith('<style rel="stylesheet" type="text/css" media="screen" data-custom-css>');
          } else {
            this.$css.replaceWith('<script type="text/css" data-custom-css>');
          }

          this.updateStoreState({use: value});
          this.$css = $('[data-custom-css]').text(text);
        }
      },

      content: function(value, oldValue) {
        if (oldValue !== null) {
          this.$css.text(value);
        }
      }
    },

    computed: _.extend(
      Vuex.mapGetters([
        'switcher',
        'content',
        'use',
      ]), {

        codeMirrorInstance: {
          cache: false,
          get: function() {
            return jQuery('[data-css-editor]').data('CodeMirror');
          }
        },

        isTextareaInitialized: {
          cache: false,
          get: function() {
            return 'undefined' !== typeof(this.codeMirrorInstance);
          }
        }
      }),

    methods: _.extend(
      Vuex.mapActions([
        'setSwitcherState',
        'updateStoreState',
        'updateOriginalState',
      ]), {

        findElements: function() {
          this.$css = $('[data-custom-css]');
        },

        enableLiveReloading: function() {
          $('body').addClass('live-css-reloading');
        },

        disableLiveReloading: function() {
          $('body').removeClass('live-css-reloading');
        },

        initTextarea: function() {
          var textarea = this.$el.querySelector('[data-css-editor]');
          var content = this.$el.querySelector('[data-css-content]');
          var text = _.unescape(content.innerHTML);
          textarea.value = text;
          this.updateOriginalState({use: this.switcher, content: text});
          this.updateStoreState({use: this.switcher, content: text});

          var self = this;
          jQuery(document).ready(function () {
            self.resizeTextarea();
            self.codeMirrorInstance.on('change', _.bind(self.onCodeMirrorChange, this));
          });
        },

        resizeTextarea: function() {
          if (this.isTextareaInitialized) {
            var width = $(this.$el).width();
            var height = $(this.$el).height();
            this.codeMirrorInstance.setSize(width, height);
          }
        },

        onCodeMirrorChange: _.debounce(function (instance) {
          this.updateStoreState({content: instance.getValue()});
        }, 300),

        onSaveSuccess: function() {
          EventBus.$emit('completed.save');
          this.updateOriginalState({use: this.use, content: this.content});
          this.updateStoreState({use: this.use, content: this.content});
        },

        onSaveFail: function() {
          EventBus.$emit('failed.save');
        },

        switchTab: function (mode) {
          EventBus.$emit('triggerSwitchTab', mode);
        },

        onSwitchChange: function (event) {
          this.$store.dispatch('toggleSwitcher', event.target.checked);
        },
      })
  });
});
