<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\Stripe\Core;

use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

/**
 * OAuth
 */
class OAuth extends \XLite\Base\Singleton
{
    /**
     * URL state
     *
     * @var string
     */
    protected $urlState;

    /**
     * Get client id
     *
     * @return string
     */
    public function getClientIdLive()
    {
        return 'ca_3M7dbaBh8wZ4nvntmkUHLl3xIbjQufNf';
    }

    /**
     * Get redirect URI
     *
     * @return string
     */
    public function getRedirectURI()
    {
        return \XLite::getInstance()->getShopURL(\XLite\Core\Converter::buildURL('stripe_oauth'));
    }

    // {{{ OAuth requests

    protected function getHttpClient(): HttpClientInterface
    {
        return HttpClient::create();
    }

    /**
     * Authorize
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     * @param string $code OAuth code
     *
     * @return array
     */
    public function authorize(\XLite\Model\Payment\Method $method, $code)
    {
        try {
            $response = $this->getHttpClient()->request(...$this->setupOAuthTokenRequest('oauth/token', $method, $code));
            $data = json_decode($response->getContent(false));
        } catch (TransportExceptionInterface) {
        }

        $result = [false, null];
        if (!empty($data->access_token)) {
            $this->saveSettings($method, $data);
            $result = [true, null];
        } elseif (!empty($data->error)) {
            $result = [false, $data->error_description];
        }

        return $result;
    }

    /**
     * Refresh token
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     *
     * @return array
     */
    public function refreshToken(\XLite\Model\Payment\Method $method)
    {
        try {
            $response = $this->getHttpClient()->request(...$this->setupOAuthTokenRefreshRequest('oauth/token', $method));
            $data = json_decode($response->getContent(false));
        } catch (TransportExceptionInterface) {
        }

        $result = [false, null];
        if (!empty($data->access_token)) {
            $this->saveSettings($method, $data, false);
            $result = [true, null];
        } elseif (!empty($data->error)) {
            $result = [false, $data->error_description];
        }

        return $result;
    }

    protected function getRequestUrl(string $path): string
    {
        return 'https://my.x-cart.com/proxy/stripe.php?path=' . $path;
    }

    protected function setupOAuthTokenRequest(string $path, \XLite\Model\Payment\Method $method, string $code): array
    {
        return [
            'POST',
            $this->getRequestUrl($path),
            [
                'body'    => [
                    'mode'          => $method->getSetting('mode'),
                    'code'          => $code,
                    'grant_type'    => 'authorization_code'
                ],
                'headers' => [
                    'Accept' => 'application/json'
                ]
            ]
        ];
    }

    protected function setupOAuthTokenRefreshRequest(string $path, \XLite\Model\Payment\Method $method): array
    {
        return [
            'POST',
            $this->getRequestUrl($path),
            [
                'body'    => [
                    'mode'          => $method->getSetting('mode'),
                    'refresh_token' => $method->getSetting('refreshToken'),
                    'grant_type'    => 'refresh_token'
                ],
                'headers' => [
                    'Accept' => 'application/json'
                ]
            ]
        ];
    }

    /**
     * Save settings
     *
     * @param \XLite\Model\Payment\Method $method Payment method
     * @param object $data OAuth response
     * @param boolean $saveFull Save full data flag OPTIONAL
     *
     * @return void
     */
    protected function saveSettings(\XLite\Model\Payment\Method $method, $data, $saveFull = true)
    {
        $prefix = $method->getProcessor()->isTestMode($method) ? 'Test' : '';
        $method->setSetting('accessToken' . $prefix, $data->access_token);
        $method->setSetting('publishKey' . $prefix, $data->stripe_publishable_key);

        if ($saveFull) {
            $method->setSetting('refreshToken', $data->refresh_token);
            $method->setSetting('userId', $data->stripe_user_id);
        }
    }

    // }}}

    // {{{ Service

    /**
     * Generate and get URL state
     *
     * @return string
     */
    public function generateURLState()
    {
        if (!isset($this->urlState)) {
            $this->urlState = $this->defineURLState();
        }

        return $this->getRedirectURI() . '|' . $this->urlState;
    }

    /**
     * Define URL state
     *
     * @return string
     */
    public function defineURLState()
    {
        return hash_hmac(
            'sha512',
            \XLite\Core\Auth::getInstance()->getProfile()->getLogin(),
            'stripe'
        );
    }

    // }}}
}
