<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\SagePay\EventListener;

use ApiPlatform\Exception\InvalidArgumentException;
use XCart\Event\Payment\PaymentActionEvent;
use XCart\Payment\RequestProcessor;
use XCart\Payment\URLGenerator\BackendURLGenerator;
use XCart\Payment\URLGenerator\URLGeneratorInterface;
use XLite\Model\Payment\Method;
use XLite\Model\Payment\Transaction;

final class PaymentProcessor
{
    private const THOUSAND_DELIMITER = ',';
    private const DECIMAL_DELIMITER  = '.';

    public function __construct(
        private BackendURLGenerator $backendURLGenerator
    ) {
    }

    public function onPaymentInitAction(PaymentActionEvent $event): void
    {
        /** @var Method $method */
        $method = $event->getMethod();

        $event->setOutputData([
            'VPSProtocol' => '4.00',
            'TxType'      => $method->getSetting('type') === 'sale'
                ? 'PAYMENT'
                : 'DEFERRED',
            'Vendor'      => $method->getSetting('vendorName'),
            'Crypt'       => $this->getCrypt($event->getTransaction(), $event->getMetaData(), $method->getSetting('password')),
        ]);

        $event->setOutputMetaData([
            'url'    => $this->getFormURL($method->getSetting('test')),
            'method' => 'post',
        ]);

        $event->setOutputCode(RequestProcessor::OUTPUT_CODE_SILENT);
    }

    public function onPaymentPayAction(PaymentActionEvent $event): void
    {
        $data = $event->getData();

        /** @var Method $method */
        $method = $event->getMethod();

        $requestBody = $this->decode($data['crypt'], $method->getSetting('password'));

        $status = RequestProcessor::OUTPUT_CODE_FAILED;

        if (isset($requestBody['Status'])) {
            if ($requestBody['Status'] === 'OK') {
                // Success status
                $event->addOutputTransactionData('TxAuthNo', $requestBody['TxAuthNo'], 'Authorisation code of the transaction');
                $status = RequestProcessor::OUTPUT_CODE_COMPLETED;
            }

            $event->addOutputTransactionData('StatusDetail', $requestBody['StatusDetail'], 'Status details');
            $event->setOutputNote($requestBody['StatusDetail']);
        } else {
            // Invalid response
            $event->addOutputTransactionData('StatusDetail', 'Invalid response was received', 'Status details');
        }

        if (isset($requestBody['VPSTxId'])) {
            $event->addOutputTransactionData('VPSTxId', $requestBody['VPSTxId'], 'The unique Opayo ID of the transaction');
        }

        if (isset($requestBody['AVSCV2'])) {
            $event->addOutputTransactionData('AVSCV2', $requestBody['AVSCV2'], 'AVSCV2 Status');
        }

        if (isset($requestBody['AddressResult'])) {
            $event->addOutputTransactionData('AddressResult', $requestBody['AddressResult'], 'Cardholder address checking status');
        }

        if (isset($requestBody['PostCodeResult'])) {
            $event->addOutputTransactionData('PostCodeResult', $requestBody['PostCodeResult'], 'Cardholder postcode checking status');
        }

        if (isset($requestBody['CV2Result'])) {
            $event->addOutputTransactionData('CV2Result', $requestBody['CV2Result'], 'CV2 code checking result');
        }

        if (isset($requestBody['3DSecureStatus'])) {
            $event->addOutputTransactionData('3DSecureStatus', $requestBody['3DSecureStatus'], '3DSecure checking status');
        }

        $total = $this->getSagePayTotal($requestBody);

        if (!$event->checkTotal($total)) {
            $event->addOutputTransactionData('StatusDetail', 'Invalid amount value was received', 'Status details');
            $status = RequestProcessor::OUTPUT_CODE_FAILED;
        }

        $event->setOutputCode($status);
    }

    private function getFormURL(bool $testMode): string
    {
        return $testMode
            ? 'https://sandbox.opayo.eu.elavon.com/gateway/service/vspform-register.vsp'
            : 'https://live.opayo.eu.elavon.com/gateway/service/vspform-register.vsp';
    }

    private function getCrypt(Transaction $transaction, array $metaData, string $password): string
    {
        $fields = $this->getOrderingInformation($transaction, $metaData);

        $cryptedFields = [];
        foreach ($fields as $key => $value) {
            $cryptedFields[] = $key . '=' . $value;
        }

        return $this->encryptAndEncode(implode('&', $cryptedFields), $password);
    }

    private function getOrderingInformation(Transaction $transaction, array $metaData): array
    {
        $currency = $transaction->getCurrency();

        $order = $transaction->getOrder();
        $profile = $order->getProfile();

        /** @var \XLite\Model\Address $billingAddress */
        $billingAddress = $profile->getBillingAddress();

        if ($billingAddress === null) {
            throw new InvalidArgumentException('Billing address is not defined');
        }

        $shippingAddress = $profile->getShippingAddress() ?: $billingAddress;

        $fields = [
            'VendorTxCode' => $transaction->getPublicId(),
            'ReferrerID'   => '653E8C42-AD93-4654-BB91-C645678FA97B',
            'Amount'       => round($transaction->getValue(), 2),
            'Currency'     => strtoupper($currency->getCode()),
            'Description'  => 'Your Cart',

            'SuccessURL' => $metaData['success_url'] ?? $this->backendURLGenerator->generateReturnURL($transaction, URLGeneratorInterface::RETURN_TXN_ID, true),
            'FailureURL' => $metaData['failure_url'] ?? $this->backendURLGenerator->generateCancelURL($transaction, URLGeneratorInterface::RETURN_TXN_ID, true),

            'CustomerName'  => $billingAddress->getFirstname()
                . ' '
                . $billingAddress->getLastname(),
            'CustomerEMail' => $profile->getLogin(),
            'VendorEMail'   => \XLite\Core\Mailer::getOrdersDepartmentMail(),
            'SendEMail'     => $this->getOptionValueSendEMail(),

            'BillingSurname'    => $billingAddress->getLastname(),
            'BillingFirstnames' => $billingAddress->getFirstname(),
            'BillingAddress1'   => $billingAddress->getAddressLineConcat(),
            'BillingCity'       => $billingAddress->getCity(),
            'BillingPostCode'   => $billingAddress->getZipcode(),
            'BillingCountry'    => strtoupper($billingAddress->getCountry()->getCode()),

            'DeliverySurname'    => $shippingAddress->getLastname(),
            'DeliveryFirstnames' => $shippingAddress->getFirstname(),
            'DeliveryAddress1'   => $shippingAddress->getAddressLineConcat(),
            'DeliveryCity'       => $shippingAddress->getCity(),
            'DeliveryPostCode'   => $shippingAddress->getZipcode(),
            'DeliveryCountry'    => strtoupper($shippingAddress->getCountry()->getCode()),

            'Basket'        => $this->getBasket(),
            'AllowGiftAid'  => 0,
            'ApplyAVSCV2'   => 0,
            'Apply3DSecure' => 0,
        ];

        if ($fields['BillingCountry'] === 'US') {
            $fields['BillingState'] = $billingAddress->getState()->getCode();
        }

        if ($fields['DeliveryCountry'] === 'US') {
            $fields['DeliveryState'] = $shippingAddress->getState()->getCode();
        }

        return $this->cropFieldsValues($fields);
    }

    private function getOptionValueSendEMail(): string
    {
        return '1';
    }

    private function getBasket(): string
    {
        return '';
    }

    private function cropFieldsValues(array $fields): array
    {
        $lengths2fields = [
            'VendorTxCode' => 40,
            'VendorEMail' => 255,
            'CustomerName' => 100,
            'CustomerEMail' => 255,
            'FailureURL' => 2000,
            'SuccessURL' => 2000,
            'BillingSurname' => 20,
            'BillingFirstnames' => 20,
            'BillingAddress1' => 100,
            'BillingCity' => 40,
            'DeliverySurname' => 20,
            'DeliveryFirstnames' => 20,
            'DeliveryAddress1' => 100,
            'DeliveryCity' => 40,
            'BillingPostCode' => 10,
        ];

        foreach ($lengths2fields as $fieldName => $length) {
            if (!empty($fields[$fieldName]) && strlen($fields[$fieldName]) > $length) {
                $fields[$fieldName] = substr($fields[$fieldName], 0, $length);
            }
        }

        return $fields;
    }

    private function decode(string $strIn, string $password): array
    {
        $sagePayResponse = [];
        $decodedString   = $this->decodeAndDecrypt($strIn, $password);
        parse_str($decodedString, $sagePayResponse);

        return $sagePayResponse;
    }

    private function decodeAndDecrypt(string $strIn, string $password): string
    {
        $aes = new \phpseclib\Crypt\AES();

        $aes->setKey($password);
        $aes->setIV($password);
        $aes->setKeyLength(128);

        return $aes->decrypt(pack('H*', substr($strIn, 1)));
    }

    private function encryptAndEncode(string $strIn, string $password): string
    {
        $aes = new \phpseclib\Crypt\AES();

        $aes->setKey($password);
        $aes->setIV($password);
        $aes->setKeyLength(128);

        return '@' . bin2hex($aes->encrypt($strIn));
    }

    /**
     * Returns calculated sage pay total to compare with order data. Excepts SagePay Surcharge.
     */
    private function getSagePayTotal(array $requestData): float
    {
        $total = $requestData['Amount'];

        if (!is_float($total)) {
            $total = (float) str_replace(
                [self::THOUSAND_DELIMITER, self::DECIMAL_DELIMITER],
                ['', '.'],
                $total
            );
        }

        $surcharge = $requestData['Surcharge'] ?? 0.0;

        if ($surcharge && !is_float($surcharge)) {
            $surcharge = (float) str_replace(
                [self::THOUSAND_DELIMITER, self::DECIMAL_DELIMITER],
                ['', '.'],
                $surcharge
            );
        }

        return $total - $surcharge;
    }
}
