<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MultiCurrency\Model;

use XCart\Extender\Mapping\Extender;
use Doctrine\ORM\Mapping as ORM;
use XLite\Core\Auth;
use XC\MultiCurrency\Core\MultiCurrency;

/**
 * Class represents an order
 * @Extender\Mixin
 */
class Order extends \XLite\Model\Order
{
    /**
     * Selected multi currency
     *
     * @var \XLite\Model\Currency
     *
     * @ORM\OneToOne (targetEntity="XLite\Model\Currency")
     * @ORM\JoinColumn (name="multi_currency_id", referencedColumnName="currency_id")
     */
    protected $selectedMultiCurrency = null;

    /**
     * Selected multi currency rate
     *
     * @var float
     *
     * @ORM\Column (type="decimal", precision=14, scale=4)
     */
    protected $selectedMultiCurrencyRate = 1.0;

    /**
     * @return void
     */
    public function updateOrder()
    {
        if (
            !isset($this->selectedMultiCurrency)
            && (
                !$this->getProfile()
                || !Auth::getInstance()->getProfile()
                || $this->getProfile()->getProfileId() === Auth::getInstance()->getProfile()->getProfileId()
            )
        ) {
            $this->updateMultiCurrency(
                MultiCurrency::getInstance()->getSelectedMultiCurrency()
            );
        }

        parent::updateOrder();
    }

    /**
     * Update multicurrency
     *
     * @param ActiveCurrency $currency
     */
    public function updateMultiCurrency(\XC\MultiCurrency\Model\ActiveCurrency $currency)
    {
        if ($currency) {
            $this->setSelectedMultiCurrency($currency->getCurrency());
            $this->setSelectedMultiCurrencyRate($currency->getRate());
        }
    }

    /**
     * Check if order is multi currency order (display currency is different from charge currency)
     *
     * @return boolean
     */
    public function isMultiCurrencyOrder()
    {
        $return = false;

        $orderCurrency = $this->getCurrency();
        $selectedCurrency = $this->getSelectedMultiCurrency();

        if (
            isset($selectedCurrency)
            && $orderCurrency->getCode() != $selectedCurrency->getCode()
        ) {
            $return = true;
        }

        return $return;
    }

    /**
     * Set selectedMultiCurrencyRate
     *
     * @param float $selectedMultiCurrencyRate
     *
     * @return Order
     */
    public function setSelectedMultiCurrencyRate($selectedMultiCurrencyRate)
    {
        $this->selectedMultiCurrencyRate = $selectedMultiCurrencyRate;
        return $this;
    }

    /**
     * Get selectedMultiCurrencyRate
     *
     * @return float
     */
    public function getSelectedMultiCurrencyRate()
    {
        return $this->selectedMultiCurrencyRate;
    }

    /**
     * Set selectedMultiCurrency
     *
     * @param \XLite\Model\Currency $selectedMultiCurrency
     *
     * @return Order
     */
    public function setSelectedMultiCurrency(\XLite\Model\Currency $selectedMultiCurrency = null)
    {
        $this->selectedMultiCurrency = $selectedMultiCurrency;
        return $this;
    }

    /**
     * Get selectedMultiCurrency
     *
     * @return \XLite\Model\Currency
     */
    public function getSelectedMultiCurrency()
    {
        return $this->selectedMultiCurrency;
    }
}
