<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MultiCurrency\Core\RateProvider;

use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use XLite\InjectLoggerTrait;

/**
 * Currency converter (https://www.currencyconverterapi.com/)
 */
class FreeCurrencyConverterAPI extends \XC\MultiCurrency\Core\RateProvider\ARateProvider
{
    use InjectLoggerTrait;

    protected HttpClientInterface $httpClient;

    public function __construct(HttpClientInterface $httpClient)
    {
        $this->httpClient = $httpClient;
    }

    /**
     * URL to post request for rate
     *
     * @var string
     */
    protected $url = 'https://free.currconv.com/api/v7/';

    /**
     * Get currency conversion rate
     *
     * @param string $from Source currency code (alpha-3)
     * @param string $to   Destination currency code (alpha-3)
     *
     * @return float
     */
    public function getRate($from, $to)
    {
        $result = null;

        $response = $this->sendRequest([
            'q'       => $from . '_' . $to,
            'compact' => 'ultra',
            'apiKey'  => $this->getApiKey(),
        ]);

        if ($response !== null) {
            $rate = $this->parseResponse($from, $to, $response);

            if ($rate) {
                $result = (float) $rate;
            }
        }

        return $result;
    }

    protected function getApiKey()
    {
        return \XLite\Core\Config::getInstance()
            ->XC
            ->MultiCurrency
            ->currency_converter_api_key;
    }

    /**
     * @param array $data
     *
     * @return null|string
     */
    protected function sendRequest(array $data)
    {
        try {
            $response = $this->httpClient->request(
                'GET',
                $this->url . 'convert?' . http_build_query($data, null, '&')
            );
            if ($response->getStatusCode() === 200 && !empty($response->getContent())) {
                $this->getLogger('XC-MultiCurrency')->debug('Response received', [
                    'data' => $response->getContent()
                ]);

                return $response->getContent();
            }
            $this->getLogger('XC-MultiCurrency')->error('Wrong response received', [
                'data' => $response->getContent(false)
            ]);
        } catch (TransportExceptionInterface $e) {
            $this->getLogger('XC-MultiCurrency')->error('API error', [
                'message' => $e->getMessage()
            ]);
        }

        return null;
    }

    /**
     * Parse server response
     *
     * @param string $from     Source currency code (alpha-3)
     * @param string $to       Destination currency code (alpha-3)
     * @param string $response Server response
     *
     * @return string
     */
    protected function parseResponse($from, $to, $response)
    {
        $q        = $from . '_' . $to;
        $response = @json_decode($response, true);

        return $response[$q] ?? null;
    }
}
