<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\View\Model\Profile;

use XCart\Extender\Mapping\Extender;
use XLite\Core\Config;
use XC\MailChimp\Core;

/**
 * @Extender\Mixin
 * @Extender\Before ("XC\TwoFactorAuthentication")
 */
abstract class Main extends \XLite\View\Model\Profile\Main
{
    /**
     * @inheritDoc
     */
    public function getCSSFiles()
    {
        return array_merge(
            parent::getCSSFiles(),
            [
                'modules/XC/MailChimp/profile/subscriptions_register.css'
            ]
        );
    }

    /**
     * Return fields list by the corresponding schema
     *
     * @return array
     */
    protected function getFormFieldsForSectionMain()
    {
        if (
            \XLite\Core\Request::getInstance()->mode == 'register'
            && $this->hasActiveMailChimpLists()
        ) {
            $additionalSchema = [
                Core\MailChimp::SUBSCRIPTION_TO_ALL_FIELD_NAME => [
                    self::SCHEMA_CLASS      => 'XLite\View\FormField\Input\Checkbox',
                    self::SCHEMA_LABEL      => static::t('Sign up for <Company name> news, sales and deals', ['company_name' => Config::getInstance()->Company->company_name]),
                    self::SCHEMA_REQUIRED   => false,
                    self::SCHEMA_IS_CHECKED => true,
                    \XLite\View\FormField\AFormField::PARAM_WRAPPER_CLASS => 'input input-checkbox mailchimp-subscribe-all not-floating',
                ]
            ];
            $schema = array_merge($this->mainSchema, $additionalSchema);

            // Modify the main schema
            $this->mainSchema = $schema;
        }

        return parent::getFormFieldsForSectionMain();
    }

    /**
     * Return MailChimp list
     *
     * @return \XC\MailChimp\Model\MailChimpList[]
     */
    protected function getMailChimpSubscriptionsList()
    {
        $profile = \XLite\Core\Auth::getInstance()->getProfile();

        $lists = \XLite\Core\Database::getRepo('\XC\MailChimp\Model\MailChimpList')
            ->getAllMailChimpLists();

        $return = [];

        foreach ($lists as $list) {
            if ($list->getEnabled()) {
                $return[] = $list;
            } elseif (
                !is_null($profile)
                && $list->isProfileSubscribed($profile)
            ) {
                $return[] = $list;
            }
        }

        return $return;
    }

    /**
     * Check if there are any active MailChimp lists
     *
     * @return boolean
     */
    protected function hasActiveMailChimpLists()
    {
        return \XLite\Core\Database::getRepo('\XC\MailChimp\Model\MailChimpList')
            ->countActiveMailChimpLists() > 0;
    }
}
