<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\View\ItemsList\Subscriptions;

use XC\MailChimp\Core;

/**
 * MailChimp mail lists
 */
abstract class AMailChimpSubscriptions extends \XLite\View\Container
{
    public const PARAM_PROFILE = 'profile';

    /**
     * Get directory
     *
     * @return string
     */
    public function getDir()
    {
        return 'modules/XC/MailChimp/profile';
    }

    /**
     * Define widget parameters
     *
     * @return void
     */
    protected function defineWidgetParams()
    {
        parent::defineWidgetParams();

        $this->widgetParams += [
            self::PARAM_PROFILE => new \XLite\Model\WidgetParam\TypeObject(
                'Profile',
                null,
                false,
                '\XLite\Model\Profile'
            )
        ];
    }

    /**
     * Get checkbox name from list
     *
     * @param \XC\MailChimp\Model\MailChimpList $list MailChimp list
     *
     * @return string
     */
    protected function getCheckboxName(\XC\MailChimp\Model\MailChimpList $list)
    {
        return Core\MailChimp::SUBSCRIPTION_FIELD_NAME . '[' . $list->getId() . ']';
    }

    /**
     * Get checkbox ID from list
     *
     * @param \XC\MailChimp\Model\MailChimpList $list MailChimp list
     *
     * @return string
     */
    protected function getCheckboxId(\XC\MailChimp\Model\MailChimpList $list)
    {
        return Core\MailChimp::SUBSCRIPTION_FIELD_NAME . '-' . $list->getId();
    }

    /**
     * Get select box ID from list
     *
     * @return string
     */
    protected function getSelectBoxId()
    {
        return Core\MailChimp::SUBSCRIPTION_FIELD_NAME;
    }

    /**
     * Get select box name from list
     *
     * @return string
     */
    protected function getSelectBoxName()
    {
        return Core\MailChimp::SUBSCRIPTION_FIELD_NAME;
    }

    /**
     * Check if display type is select box
     *
     * @return boolean
     */
    protected function isSelectBoxElement()
    {
        return Core\MailChimp::isSelectBoxElement();
    }

    /**
     * Return MailChimp list
     *
     * @param boolean $countOnly Return only number of lists OPTIONAL
     *
     * @return \XC\MailChimp\Model\MailChimpList[] | integer
     */
    protected function getData($countOnly = false)
    {
        $return = [];

        $profile = $this->getProfile();

        $lists = \XLite\Core\Database::getRepo('\XC\MailChimp\Model\MailChimpList');

        if ($countOnly) {
            $return = $lists->countActiveMailChimpLists();
        } else {
            $lists = $lists->getAllMailChimpLists();

            foreach ($lists as $list) {
                if ($list->getEnabled()) {
                    $return[] = $list;
                } elseif ($list->isProfileSubscribed($profile)) {
                    $return[] = $list;
                }
            }
        }

        return $return;
    }

    /**
     * Get current profile
     *
     * @return \XLite\Model\Profile
     */
    protected function getProfile()
    {
        return $this->getParam(self::PARAM_PROFILE);
    }

    /**
     * @return \Doctrine\ORM\PersistentCollection
     */
    public function getGroupsForSelectBox()
    {
        $cnd = new \XLite\Core\CommonCell();

        $cnd->enabled = \XLite\Model\SearchCondition\Expression\TypeEquality::create(
            'enabled',
            true
        );
        $cnd->listEnabled = \XLite\Model\SearchCondition\Expression\TypeEquality::create(
            'list.enabled',
            true
        );

        return \XLite\Core\Database::getRepo('\XC\MailChimp\Model\MailChimpGroup')
            ->search($cnd);
    }
}
