<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\Model;

use XCart\Extender\Mapping\Extender;
use XLite\InjectLoggerTrait;
use XC\MailChimp\Core;
use XC\MailChimp\Main;

/**
 * Class represents an order
 * @Extender\Mixin
 */
abstract class Cart extends \XLite\Model\Cart
{
    use InjectLoggerTrait;

    protected static $mcNewCartFlag;

    /**
     * Method to access a singleton
     *
     * @param boolean $doCalculate Flag for cart recalculation OPTIONAL
     *
     * @return \XLite\Model\Cart
     */
    public static function getInstance($doCalculate = true)
    {
        $cart = parent::getInstance($doCalculate);

        if (!isset(static::$mcNewCartFlag)) {
            static::$mcNewCartFlag = !$cart->isPersistent();
        }

        return $cart;
    }

    /**
     * Check if mc should create new cart without checking
     *
     * @return boolean
     */
    public static function isMcNewCart()
    {
        return static::$mcNewCartFlag;
    }

    /**
     * Additional check for cart update action
     *
     * @return bool
     */
    public function isUpdateActionAllowed()
    {
        return true;
    }

    /**
     * Called when an order successfully placed by a client
     *
     * @return void
     */
    public function processSucceed()
    {
        parent::processSucceed();

        /** @see \XC\MailChimp\Model\Order::isECommerce360Order() */
        if (
            ($this->isECommerce360Order()
                || Main::getStoreForDefaultAutomation())
            && Main::isMailChimpECommerceConfigured()
        ) {
            try {
                $mcCore = Core\MailChimp::getInstance();

                $mcCore->createOrder($this);

                $mcCore->removeCart($this);
            } catch (\Exception $e) {
                $this->getLogger('XC-MailChimp')->debug($e->getMessage());
            }
        }

        $profile = $this->getAvailableProfile();
        if (
            $profile
            && $profile->hasMailChimpSubscriptions()
        ) {
            $profile->checkSegmentsConditions();
        }
    }

    /**
     * @return bool
     */
    public function needUpdateMailchimpCart()
    {
        $request = \XLite\Core\Request::getInstance();
        $result  = true;

        if (
            $request->widget === '\XLite\View\Minicart'
            || ($request->target === 'checkout'
                && $request->action !== 'shipping'
            )
        ) {
            $result = false;
        }

        return $result;
    }

    /**
     * Get available profile
     *
     * @return \XC\MailChimp\Model\Profile|\XLite\Model\Profile
     */
    protected function getAvailableProfile(): \XLite\Model\Profile
    {
        return $this->getOrigProfile() ? $this->getOrigProfile() : $this->getProfile();
    }
}
