<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp;

use XLite\Core\Auth;
use XLite\Core\Config;

abstract class Main extends \XLite\Module\AModule
{
    /**
     * Return link to settings form
     *
     * @return string
     */
    public static function getSettingsForm()
    {
        return \XLite\Core\Converter::buildURL('mailchimp_options');
    }

    public static function updateAllMainStores()
    {
        $stores = static::getMainStores();

        $ecCore = \XC\MailChimp\Core\MailChimpECommerce::getInstance();

        foreach ($stores as $store) {
            $ecCore->updateStoreAndReference($store->getList()->getId(), true);
        }

        $ecCore->updateConnectedSites();

        \XLite\Core\Database::getEM()->flush();
    }

    public static function setAllStoreSyncFlag($flagValue)
    {
        $stores = static::getMainStores();

        $ecCore = \XC\MailChimp\Core\MailChimpECommerce::getInstance();

        foreach ($stores as $store) {
            $ecCore->changeStoreSyncingStatus($store->getId(), $flagValue);
        }
    }

    /**
     * @return \XC\MailChimp\Model\Store
     */
    public static function getStoreForDefaultAutomation()
    {
        /** @var \XC\MailChimp\Model\Repo\Store $repo */
        $repo = \XLite\Core\Database::getRepo('XC\MailChimp\Model\Store');

        return $repo->findOneByList(
            \XLite\Core\Config::getInstance()->XC->MailChimp->defaultAutomationListId
        );
    }

    /**
     * @return \XC\MailChimp\Model\Store[]
     */
    public static function getMainStores()
    {
        /** @var \XC\MailChimp\Model\Repo\Store $repo */
        $repo = \XLite\Core\Database::getRepo('XC\MailChimp\Model\Store');

        return $repo->findBy([
            'main' => true,
        ]);
    }

    /**
     * Check if MailChimp module is configured and have lists
     *
     * @return boolean
     */
    public static function isMailChimpConfigured()
    {
        $listsRepo = \XLite\Core\Database::getRepo('\XC\MailChimp\Model\MailChimpList');

        return \XC\MailChimp\Core\MailChimp::hasAPIKey()
               && 0 < $listsRepo->countActiveMailChimpLists();
    }

    /**
     * Check if MailChimp module is configured, have lists and have ecommerce enabled
     *
     * @return boolean
     */
    public static function isMailChimpECommerceConfigured()
    {
        return static::isMailChimpConfigured()
               && \XLite\Core\Config::getInstance()->XC->MailChimp->analytics360enabled;
    }

    /**
     * Check if MailChimp module is configured, have lists,
     * have e-commerce enabled and have abandonedCart enabled
     *
     * @return bool
     */
    public static function isMailChimpAbandonedCartEnabled()
    {
        return static::isMailChimpECommerceConfigured()
               && \XLite\Core\Config::getInstance()->XC->MailChimp->abandonedCartEnabled;
    }

    /**
     * @return bool
     */
    public static function isDebugMode()
    {
        return (bool) $_ENV['APP_DEBUG'];
    }

    /**
     * @return boolean
     */
    public static function hasGdprRelatedActivity()
    {
        return true;
    }

    public static function getShowForAdmins(): array
    {
        $showForAdmins = Config::getInstance()->XC->MailChimp->show_for_admins;
        if (!empty($showForAdmins)) {
            $showForAdmins = \json_decode($showForAdmins, true);
        }

        return $showForAdmins ?: [];
    }

    public static function getCacheParameters(): array
    {
        sort($showForAdmins = static::getShowForAdmins());

        return [md5(implode(',', $showForAdmins))];
    }

    public static function labelIsShown()
    {
        $showForAdmins = static::getShowForAdmins();

        if ($showForAdmins) {
            $profileId = Auth::getInstance()->getProfile()->getProfileId();

            return in_array($profileId, $showForAdmins, true);
        }

        return false;
    }
}
