<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\EntityCallback;

use Doctrine\Persistence\Event\LifecycleEventArgs;
use XC\MailChimp\Core\Request\Product\Create;
use XC\MailChimp\Core\Request\Product\Remove;
use XC\MailChimp\Main;
use XC\MailChimp\Logic\DataMapper;
use XC\MailChimp\Core\Action\ProductUpdate;

final class ProductCallback
{
    public function postPersist(\XLite\Model\Product $product, LifecycleEventArgs $args)
    {
        if (!$product->getProductId()) {
            $product->setProductId($args->getObject()->getProductId());
        }

        if (Main::isMailChimpECommerceConfigured() && Main::getMainStores()) {
            foreach (Main::getMainStores() as $store) {
                Create::executeAction($store->getId(), DataMapper\Product::getDataByProduct($product));
            }
        }
    }

    public function preUpdate(\XLite\Model\Product $product, LifecycleEventArgs $args)
    {
        $changeSet = $args->getObjectManager()->getUnitOfWork()->getEntityChangeSet($product);
        $excludedFields = $product->getExcludedFieldsForMailchimpUpdate();

        if ($excludedFields) {
            foreach ($excludedFields as $field) {
                if (isset($changeSet[$field])) {
                    unset($changeSet[$field]);
                }
            }
        }

        if (
            Main::isMailChimpECommerceConfigured() && Main::getMainStores()
            && $product->getProductId()
            && (array_filter($changeSet) || empty($changeSet))
        ) {
            $action = new ProductUpdate($product);
            $action->execute();
        }
    }

    public function preRemove(\XLite\Model\Product $product, LifecycleEventArgs $args)
    {
        if (Main::isMailChimpECommerceConfigured() && Main::getMainStores()) {
            foreach (Main::getMainStores() as $store) {
                Remove::executeAction($store->getId(), $product->getProductId());
            }
        }
    }
}
