<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\Core\Request\Store;

use XC\MailChimp\Core\Request\MailChimpRequest;
use XC\MailChimp\Model\Store;

class Update extends MailChimpRequest
{
    /**
     * @param array $data
     */
    public function __construct($data)
    {
        $this->data = $data;

        parent::__construct('Updating store', 'patch', "ecommerce/stores/{$this->data['storeId']}", $this->data['storeData']);
    }

    /**
     * @param string $storeId
     * @param array  $storeData
     *
     * @return self
     */
    public static function getRequest($storeId, $storeData): self
    {
        return new self([
            'storeId' => $storeId,
            'storeData' => $storeData
        ]);
    }

    /**
     * @param string $storeId
     * @param array  $storeData
     *
     * @return void
     */
    public static function scheduleAction($storeId, $storeData)
    {
        self::getRequest($storeId, $storeData)->queue();
    }

    /**
     * @param string $storeId
     * @param array  $storeData
     *
     * @return bool
     */
    public static function updateStoreInDB($storeId, $storeData): bool
    {
        if (!Check::isStoreExistsInDB($storeId)) {
            return Create::createStoreInDB($storeData);
        }

        if (!self::canUpdateInDB($storeData)) {
            return false;
        }

        try {
            /** @var Store $store */
            $store = \XLite\Core\Database::getRepo(Store::class)->find($storeId);

            $store->setName($storeData['store_name']);
            $store->setMain($storeData['is_main'] ?? false);

            $store->update();
        } catch (\Exception $e) {
        }

        return true;
    }

    /**
     * @param $storeData
     *
     * @return bool
     */
    protected static function canUpdateInDB($storeData): bool
    {
        return isset($storeData['store_name']);
    }

    /**
     * @return array|null
     */
    public function execute(): ?array
    {
        if (Check::executeAction($this->data['storeId'])) {
            $result = parent::execute();

            self::updateStoreInDb($this->data['storeId'], $this->getArgs());

            return $result;
        }

        return null;
    }
}
