<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\Core\Request\Order;

use XLite\Model\Order;
use XC\MailChimp\Core\Request\Campaign as MailChimpCampaign;
use XC\MailChimp\Core\Request\Customer as MailChimpCustomer;
use XC\MailChimp\Core\Request\MailChimpBatchRequest;
use XC\MailChimp\Core\Request\MailChimpRequest;
use XC\MailChimp\Core\Request\Product as MailChimpProduct;
use XC\MailChimp\Core\Request\Store as MailChimpStore;
use XC\MailChimp\Logic\DataMapper;

class Create extends MailChimpRequest
{
    protected $customerId;

    protected $userIdFromRequest;

    protected $productsData = [];

    /**
     * @param array $data
     */
    public function __construct($data)
    {
        $this->data = $data;

        $order = $this->getOrder($this->data['orderId']);

        $campaignIdFromRequest = MailChimpCampaign\Get::getCampaignIdFromRequest();
        $userIdFromRequest     = MailChimpCustomer\Get::getUserIdFromRequest();
        $trackingIdFromRequest = MailChimpRequest::getTrackingIdFromRequest();

        $this->customerId        = $order->getOrigProfile() && $order->getOrigProfile()->getProfileId()
            ? $order->getOrigProfile()->getProfileId()
            : null;
        $this->userIdFromRequest = $userIdFromRequest;
        if (
            !$this->customerId
            && !$this->userIdFromRequest
            && $order->getProfile() && $order->getProfile()->getProfileId()
        ) {
            // fallback to anonymous profile
            $this->customerId = $order->getProfile()->getProfileId();
        }

        $orderData = DataMapper\Order::getDataByOrder($campaignIdFromRequest, $trackingIdFromRequest, $order);

        parent::__construct('Creating order', 'post', '', $orderData);

        $this->productsData = array_map(static function ($item) {
            return DataMapper\Product::getDataByOrderItem($item);
        }, $order->getItems()->toArray());
    }

    /**
     * @param int $orderId
     *
     * @return self
     */
    public static function getRequest($orderId): self
    {
        return new self([
            'orderId' => $orderId
        ]);
    }

    /**
     * @param Order $order
     *
     * @return void
     */
    public static function scheduleAction($orderId)
    {
        self::getRequest($orderId)->queue();
    }

    /**
     * @return array|null
     */
    public function execute(): ?array
    {
        $orderData = $this->getArgs();

        $completeCustomerData = null;

        foreach (MailChimpStore\Get::getActiveStores($orderData['campaign_id'] ?? null) as $storeId => $storeData) {
            $batchRequest = MailChimpBatchRequest::getRequest(array_map(static function ($item) use ($storeId) {
                return MailChimpProduct\Create::getRequest($storeId, $item);
            }, $this->productsData));


            $customerExists = ($this->customerId ?: $this->userIdFromRequest) && MailChimpCustomer\Check::executeAction($storeId, $this->customerId ?: $this->userIdFromRequest);

            if ($customerExists) {
                $customerData = ['id' => (string)($this->customerId ?: $this->userIdFromRequest)];
            } else {
                if ($completeCustomerData === null) {
                    $profile              = $this->getProfileByCustomerId($this->customerId);
                    $completeCustomerData = DataMapper\Customer::getDataForOrder($this->userIdFromRequest, $profile, false);
                }

                $customerData = $completeCustomerData;
            }

            $orderData['customer'] = $customerData;

            MailChimpStore\Create::executeAction([
                'id'            => $storeId,
                'name'          => $storeData['storeName'],
                'list_id'       => $storeData['listId'],
                'currency_code' => $orderData['currency_code'],
                'money_format'  => \XLite::getInstance()->getCurrency()->getPrefix() ?: \XLite::getInstance()->getCurrency()->getSuffix(),
            ]);

            $this->setStoreIdToOrder($this->data['orderId'], $storeId);

            $this->setAction("ecommerce/stores/{$storeId}/orders");
            $this->setArgs($orderData);

            $batchRequest->addOperation($this);

            MailChimpCustomer\Check::dropActionCache($storeId, $orderData['customer']['id']);

            $batchRequest->execute();
        }

        return null;
    }

    /**
     * @param string $orderId
     * @param string $storeId
     */
    protected function setStoreIdToOrder($orderId, $storeId): void
    {
        /** @var \XC\MailChimp\Model\Order|Order $order */
        $order = $this->getOrder($orderId);

        if ($order) {
            $order->setMailchimpStoreId($storeId);
        }
    }

    /**
     * @param string $orderId
     *
     * @return Order
     */
    protected function getOrder($orderId): Order
    {
        return \XLite\Core\Database::getRepo(Order::class)->find($orderId);
    }

    /**
     * @param string $customerId
     *
     * @return \XLite\Model\Profile|null
     */
    protected function getProfileByCustomerId($customerId)
    {
        /** @var \XLite\Model\Profile $profile */
        $profile = \XLite\Core\Database::getRepo('XLite\Model\Profile')->find($customerId);

        return $profile;
    }
}
