<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\Core\Request\Cart;

use XC\MailChimp\Core\Request\Customer as MailChimpCustomer;
use XC\MailChimp\Core\Request\MailChimpBatchRequest;
use XC\MailChimp\Core\Request\MailChimpRequest;
use XC\MailChimp\Core\Request\Product as MailChimpProduct;
use XC\MailChimp\Core\Request\Store as MailChimpStore;
use XC\MailChimp\Logic\DataMapper;

class UpdateOrCreate extends MailChimpRequest
{
    protected $customerId;

    protected $userIdFromRequest;

    protected $productsData = [];

    /**
     * @param array $data
     */
    public function __construct($data)
    {
        $this->data = $data;

        $this->customerId        = $this->data['profileId'];
        $this->userIdFromRequest = MailChimpCustomer\Get::getUserIdFromRequest();
        $this->productsData      = $this->data['productsData'];

        parent::__construct('', '', '', $this->data['cartData']);
    }

    /**
     * @param int   $profileId
     * @param array $cartData
     * @param array $productsData
     * @return self
     */
    public static function getRequest($profileId, $cartData, $productsData): self
    {
        return new self([
            'profileId' => $profileId,
            'cartData' => $cartData,
            'productsData' => $productsData
        ]);
    }

    /**
     * @param int   $profileId
     * @param array $cartData
     * @param array $productsData
     *
     * @return void
     */
    public static function scheduleAction($profileId, $cartData, $productsData)
    {
        self::getRequest($profileId, $cartData, $productsData)->queue();
    }

    /**
     * @param string $profileId
     * @param array  $cartData
     * @param array  $productsData
     *
     * @return mixed
     */
    public static function executeAction($profileId, $cartData, $productsData)
    {
        return self::getRequest($profileId, $cartData, $productsData)->execute();
    }

    /**
     * @return array|null
     */
    public function execute(): ?array
    {
        $cartData = $this->getArgs();

        $completeCustomerData = null;

        foreach (MailChimpStore\Get::getActiveStores($cartData['campaign_id'] ?? null) as $storeId => $storeData) {
            $batchRequest = MailChimpBatchRequest::getRequest(array_map(static function ($item) use ($storeId) {
                return MailChimpProduct\Create::getRequest($storeId, $item);
            }, $this->productsData));

            $customerExists = ($this->customerId ?: $this->userIdFromRequest) && MailChimpCustomer\Check::executeAction($storeId, $this->customerId ?: $this->userIdFromRequest);

            if ($customerExists) {
                $customerData = ['id' => (string)($this->customerId ?: $this->userIdFromRequest)];
            } else {
                if ($completeCustomerData === null) {
                    $profile              = $this->getProfileByCustomerId($this->customerId);
                    $completeCustomerData = DataMapper\Customer::getDataForOrder($this->userIdFromRequest, $profile, false);
                }

                $customerData = $completeCustomerData;
            }

            $cartData['customer'] = $customerData;

            if (!Check::executeAction($storeId, $cartData['id'])) {
                MailChimpStore\Create::executeAction([
                    'id'            => $storeId,
                    'name'          => $storeData['storeName'],
                    'list_id'       => $storeData['listId'],
                    'currency_code' => $cartData['currency_code'],
                    'money_format'  => \XLite::getInstance()->getCurrency()->getPrefix() ?: \XLite::getInstance()->getCurrency()->getSuffix(),
                ]);

                Check::dropActionCache($storeId, $cartData['id']);
                MailChimpCustomer\Check::dropActionCache($storeId, $cartData['customer']['id']);

                $this->setName('Creating cart');
                $this->setMethod('post');
                $this->setAction("ecommerce/stores/{$storeId}/carts");
            } else {
                $this->setName('Updating cart');
                $this->setMethod('patch');
                $this->setAction("ecommerce/stores/{$storeId}/carts/{$cartData['id']}");
            }

            $this->setArgs($cartData);

            $batchRequest->addOperation($this);

            $batchRequest->execute();
        }

        return null;
    }

    /**
     * @param string $customerId
     *
     * @return \XLite\Model\Profile|null
     */
    protected function getProfileByCustomerId($customerId)
    {
        /** @var \XLite\Model\Profile $profile */
        $profile = \XLite\Core\Database::getRepo('XLite\Model\Profile')->find($customerId);

        return $profile;
    }
}
