<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\Core;

use Symfony\Contracts\HttpClient\Exception\TransportExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;

class OAuth
{
    protected $clientId;
    protected $clientSecret;
    protected $oauthProxyUrl;

    protected $client;

    protected HttpClientInterface $httpClient;

    public const AUTH_ENDPOINT     = 'https://login.mailchimp.com/oauth2/authorize';
    public const TOKEN_ENDPOINT    = 'https://login.mailchimp.com/oauth2/token';
    public const METADATA_ENDPOINT = 'https://login.mailchimp.com/oauth2/metadata';

    public const GRANT_TYPE        = 'authorization_code';

    /**
     * OAuth constructor.
     *
     * @param $clientId
     * @param $clientSecret
     * @param $oauthProxyUrl
     * @param HttpClientInterface $httpClient
     */
    public function __construct($clientId, $clientSecret, $oauthProxyUrl, HttpClientInterface $httpClient)
    {
        $this->clientId     = $clientId;
        $this->clientSecret = $clientSecret;
        $this->oauthProxyUrl = $oauthProxyUrl;

        $this->client = new \OAuth2\Client($this->clientId, $this->clientSecret);
        $this->httpClient = $httpClient;
    }

    /**
     * @param $redirectUrl
     *
     * @return string
     */
    public function getAuthUrl($redirectUrl)
    {
        $realRedirectUrl = $this->wrapRedirectUrlWithProxy($redirectUrl);

        return $this->client->getAuthenticationUrl(static::AUTH_ENDPOINT, $realRedirectUrl);
    }

    /**
     * @param $code
     * @param $redirectUrl
     *
     * @return null
     */
    public function getToken($code, $redirectUrl)
    {
        $realRedirectUrl = $this->wrapRedirectUrlWithProxy($redirectUrl);

        $params = [
            'code'          => $code,
            'redirect_uri'  => $realRedirectUrl,
        ];

        $response = $this->client->getAccessToken(static::TOKEN_ENDPOINT, static::GRANT_TYPE, $params);

        return $response['result']['access_token'] ?? null;
    }

    /**
     * @param $redirectUrl
     *
     * @return string
     */
    protected function wrapRedirectUrlWithProxy($redirectUrl)
    {
        $proxyParams = [ 'redirecturl' => $redirectUrl ];
        $separator = strpos($this->oauthProxyUrl, '?') === false ? '?' : '&';

        return $this->oauthProxyUrl . $separator . http_build_query($proxyParams, null, '&');
    }

    /**
     * @param $token
     *
     * @return mixed
     * @throws \Exception
     */
    public function getTokenMetadata($token)
    {
        $responseContent = null;

        try {
            $response = $this->httpClient->request(
                'GET',
                static::METADATA_ENDPOINT,
                [
                    'headers' => [
                        'Authorization' => 'OAuth ' . $token
                    ]
                ]
            );
            $responseContent = json_decode($response->getContent(false));
        } catch (TransportExceptionInterface) {
        }

        if (!$responseContent) {
            throw new \Exception('Token metadata unavailable');
        }

        return $responseContent;
    }
}
