<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\Controller\Admin;

use XC\MailChimp\Core\MailChimpECommerce;
use XC\MailChimp\Logic\UploadingData;
use XC\MailChimp\Main;
use XLite\Controller\TitleFromController;

/**
 * Class MailchimpStoreData
 */
class MailchimpStoreData extends \XLite\Controller\Admin\AAdmin
{
    /**
     * @return void
     */
    public function handleRequest()
    {
        if (!\XC\MailChimp\Core\MailChimp::hasAPIKey()) {
            $this->redirect($this->buildURL('mailchimp_options'));
        } else {
            parent::handleRequest();
        }
    }

    /**
     * @return string
     */
    public function getTitle()
    {
        return new TitleFromController('mailchimp_options');
    }

    /**
     * Check - export process is not-finished or not
     *
     * @return bool
     */
    public function isCheckProcessNotFinished()
    {
        $eventName = UploadingData\Generator::getEventName();
        $state     = \XLite\Core\Database::getRepo(\XLite\Model\TmpVar::class)->getEventState($eventName);

        return $state
            && in_array(
                $state['state'],
                [\XLite\Core\EventTask::STATE_STANDBY, \XLite\Core\EventTask::STATE_IN_PROGRESS],
                true
            )
            && !\XLite\Core\Database::getRepo(\XLite\Model\TmpVar::class)->getVar(
                UploadingData\Generator::getCancelFlagVarName()
            );
    }

    /**
     * @inheritDoc
     */
    public static function defineFreeFormIdActions()
    {
        return array_merge(
            parent::defineFreeFormIdActions(),
            [
                'startUploadProducts',
                'startUploadOrders',
                'startUploadAll',
                'updateStoresData',
            ]
        );
    }

    public function doActionUpdateStores()
    {
        $lists = \XLite\Core\Request::getInstance()->lists ?: [];

        foreach ($lists as $listId => $value) {
            MailChimpECommerce::getInstance()->updateStoreAndReference($listId, $value);
        }

        MailChimpECommerce::getInstance()->updateConnectedSites();

        \XLite\Core\Database::getEM()->flush();
    }

    public function doActionUpdateStoresData()
    {
        Main::updateAllMainStores();

        Main::setAllStoreSyncFlag(false);
    }

    /**
     * Export action
     */
    protected function doActionStartUploadAll()
    {
        $lists = \XLite\Core\Request::getInstance()->lists;

        UploadingData\Generator::run(
            [
                'lists' => $lists,
            ]
        );
    }

    /**
     * Export action
     */
    protected function doActionStartUploadProducts()
    {
        $lists = \XLite\Core\Request::getInstance()->lists;

        UploadingData\Generator::run(
            [
                'steps' => [
                    'products',
                ],
                'lists' => $lists,
            ]
        );
    }

    /**
     * Export action
     */
    protected function doActionStartUploadOrders()
    {
        $lists = \XLite\Core\Request::getInstance()->lists;

        UploadingData\Generator::run(
            [
                'steps' => [
                    'orders',
                ],
                'lists' => $lists,
            ]
        );
    }

    /**
     * Cancel
     */
    protected function doActionCancel()
    {
        UploadingData\Generator::cancel();
        \XLite\Core\TopMessage::addWarning('Uploading data has been stopped.');

        $this->setReturnURL($this->buildURL('mailchimp_store_data'));
    }

    /**
     * Preprocessor for no-action run
     */
    protected function doNoAction()
    {
        $request = \XLite\Core\Request::getInstance();

        if ($request->process_completed) {
            \XLite\Core\TopMessage::addInfo('Uploading data has been completed successfully.');

            $this->setReturnURL(
                $this->buildURL('mailchimp_store_data')
            );
        } elseif ($request->process_failed) {
            \XLite\Core\TopMessage::addError('Uploading data has been stopped.');

            $this->setReturnURL(
                $this->buildURL('mailchimp_store_data')
            );
        }
    }
}
