<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\Controller\Admin;

use XCart\Domain\ModuleManagerDomain;
use XLite\Core\HTTP\Request;

/**
 * Sign Up modal controller
 */
class MailchimpSignUp extends \XLite\Controller\Admin\AAdmin
{
    public const LOGIN_ENDPOINT = 'https://mailchimp.xcart.com/platformapi/';

    protected ?ModuleManagerDomain $moduleManagerDomain;

    public function __construct(array $params = [])
    {
        $this->moduleManagerDomain = \XCart\Container::getContainer()?->get(ModuleManagerDomain::class);

        parent::__construct($params);
    }

    /**
     * hide default title
     *
     * @return boolean
     */
    public function isTitleVisible()
    {
        return false;
    }

    /**
     * Return the current page title (for the content area)
     *
     * @param bool $isLastStep
     *
     * @return string
     */
    public function getCustomTitle($isLastStep = false)
    {
        return static::t('Sign up for Mailchimp ({{step}} of {{count}})', [
            'step' => $isLastStep ? 2 : 1,
            'count' => 2
        ]);
    }

    /**
     * @return bool
     */
    public function isLastStep()
    {
        return \XLite\Core\Request::getInstance()->step ?? false;
    }

    /**
     * @return string
     */
    public function getEmail()
    {
        return \XLite\Core\Auth::getInstance()->getProfile()->getEmail();
    }

    /**
     * @return void
     */
    protected function doActionRegister()
    {
        $formData = \XLite\Core\Request::getInstance()->register;
        if (!empty($formData['step_first'])) {
            $this->checkAccountAvailability($formData);
        } else {
            $this->signUpRequest($formData);
        }
    }

    /**
     * @param array $formData
     *
     * @return bool
     */
    protected function checkAccountAvailability($formData)
    {
        $payload = ['username' => $formData['username']];
        $request = new Request(self::LOGIN_ENDPOINT . 'account-availability');

        $request->setHeader('Content-Type', 'application/json');
        $request->verb = 'POST';
        $request->body = json_encode($payload);

        $response = $request->sendRequest();
        if ($response && $response->code === 200) {
            $data = json_decode($response->body, true);
            if (!empty($data['errors'])) {
                $this->doInvalidField($data['errors'], $formData['step_first']);

                return false;
            }

            $this->getLogger('XC-MailChimp')->debug('Account availability check success', ['data' => $payload]);
            if (isset($data['account_available'])) {
                if ($data['account_available']) {
                    \XLite\Core\Event::switchNextStep();
                } else {
                    \XLite\Core\Event::invalidElement(
                        'register[username]',
                        static::t('Someone already has this username. Please try another username.')
                    );
                }

                return true;
            }

            $this->getLogger('XC-MailChimp')->error('Generate SignUp link error', ['data' => $data]);
            \XLite\Core\TopMessage::addError('MailChimp response error');

            return false;
        } else {
            $this->getLogger('XC-MailChimp')->error('MailChimp proxy error', ['code' => $response->code]);
            \XLite\Core\TopMessage::addError('MailChimp proxy error');

            return false;
        }
    }

    /**
     * @param array $formData
     *
     * @return bool
     */
    protected function signUpRequest($formData)
    {
        $payload = $this->getSignUpData($formData);
        $request = new Request(self::LOGIN_ENDPOINT . 'signup');

        $request->setHeader('Content-Type', 'application/json');
        $request->verb = 'POST';
        $request->body = json_encode($payload);

        $response = $request->sendRequest();
        if ($response && $response->code === 200) {
            $data = json_decode($response->body, true);
            if (!empty($data['errors'])) {
                $this->doInvalidField($data['errors'], $formData['step_first']);

                return false;
            }

            $this->getLogger('XC-MailChimp')->debug('Sign up success', ['data' => $payload]);
            if (!empty($data['oauth_token'])) {
                \XLite\Core\Database::getRepo('XLite\Model\Config')->createOption([
                    'category' => 'XC\MailChimp',
                    'name'     => 'mailChimpAPIKey',
                    'value'    => $data['oauth_token'] . '-' . $data['dc'],
                ]);

                \XLite\Core\Event::reloadOnSuccess();

                return true;
            }

            $this->getLogger('XC-MailChimp')->error('Sign up error', ['data' => $data]);
            \XLite\Core\TopMessage::addError('MailChimp sign up response error');

            return false;
        } else {
            $this->getLogger('XC-MailChimp')->error('MailChimp proxy error', ['code' => $response->code]);
            \XLite\Core\TopMessage::addError('MailChimp proxy error');

            return false;
        }
    }

    /**
     * @param array $errorData
     * @param bool $isFirstStep
     *
     * @return void
     */
    protected function doInvalidField($errorData, $isFirstStep)
    {
        $this->getLogger('XC-MailChimp')->error('MailChimp connect error', ['data' => $errorData]);

        $error = reset($errorData);
        if (
            !$isFirstStep
            && in_array($error['field'], ['username', 'email', 'fname', 'lname'])
        ) {
            \XLite\Core\Event::switchPrevStep();
        }

        \XLite\Core\Event::invalidElement(
            'register[' . str_replace('address.', '', $error['field']) . ']',
            $error['message']
        );
    }

    /**
     * @param array $formData
     *
     * @return array
     */
    protected function getSignUpData($formData): array
    {
        $module = $this->moduleManagerDomain->getModule('XC-MailChimp');

        $stateId = \XLite\Core\Request::getInstance()->address_state;
        $stateCode = \XLite\Core\Database::getRepo('XLite\Model\State')->getCodeById($stateId);
        $company = \XLite\Core\Config::getInstance()->Company;

        return [
            'email'       => $formData['email'],
            'fname'       => $formData['fname'],
            'lname'       => $formData['lname'],
            'username'    => $formData['username'],
            'app_version' => $module['version'],
            'ip_address'  => \XLite\Core\Request::getInstance()->getClientIp(),
            'timezone'    => \XLite\Core\Converter::getTimeZone() ? \XLite\Core\Converter::getTimeZone()->getName() : '',
            'org'         => ['name' => $company->company_name],
            'address'     => [
                'addr1'   => $formData['addr1'],
                'country' => $formData['country'],
                'state'   => $stateCode,
                'city'    => $formData['city'],
                'zip'     => $formData['zip'],
            ],
        ];
    }
}
