<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\MailChimp\API\Endpoint\Profile\Transformer;

use ApiPlatform\Core\Exception\InvalidArgumentException;
use ApiPlatform\Core\Serializer\AbstractItemNormalizer;
use Doctrine\Common\Collections\Collection;
use XC\MailChimp\Model\MailChimpGroupName;
use XC\MailChimp\Model\MailChimpList;
use XC\MailChimp\Model\MailChimpSegment;
use XCart\Extender\Mapping\Extender;
use XLite\API\Endpoint\Profile\DTO\ProfileInput as InputDTO;
use XLite\API\Endpoint\Profile\Transformer\InputTransformer as ExtendedInputTransformer;
use XC\MailChimp\API\Endpoint\Profile\DTO\ProfileInput as ModuleInputDTO;
use XC\MailChimp\API\Endpoint\Profile\DTO\ProfileOutput as ModuleOutputDTO;
use XC\MailChimp\Model\Profile as Model;
use XLite\Model\Profile as BaseModel;

/**
 * @Extender\Mixin
 */
class InputTransformer extends ExtendedInputTransformer
{
    /**
     * @param ModuleInputDTO $object
     */
    public function transform($object, string $to, array $context = []): BaseModel
    {
        /** @var Model $entity */
        $entity = parent::transform($object, $to, $context);

        $this->updateMailChimpList($entity->getMailChimpLists(), $object->mail_chimp_lists);
        $this->updateMailChimpGroupName($entity->getMailChimpInterests(), $object->mail_chimp_interests);
        $this->updateMailChimpList($entity->getMailChimpSegments(), $object->mail_chimp_segments);

        return $entity;
    }

    public function initialize(string $inputClass, array $context = [])
    {
        /** @var Model $entity */
        $entity = $context[AbstractItemNormalizer::OBJECT_TO_POPULATE] ?? null;
        if (!$entity) {
            return new InputDTO();
        }

        /** @var ModuleOutputDTO $input */
        $input = parent::initialize($inputClass, $context);

        $input->mail_chimp_lists = [];
        /** @var MailChimpList $subEntity */
        foreach ($entity->getMailChimpLists() as $subEntity) {
            $input->mail_chimp_lists[] = $subEntity->getId();
        }

        $input->mail_chimp_interests = [];
        /** @var MailChimpGroupName $subEntity */
        foreach ($entity->getMailChimpInterests() as $subEntity) {
            $input->mail_chimp_interests[] = $subEntity->getId();
        }
        $input->mail_chimp_segments = [];
        /** @var MailChimpSegment $subEntity */
        foreach ($entity->getMailChimpSegments() as $subEntity) {
            $input->mail_chimp_interests[] = $subEntity->getId();
        }

        return $input;
    }

    public function updateMailChimpList(Collection $collection, array $idList): void
    {
        foreach ($idList as $id) {
            $found = false;
            /** @var MailChimpList $subEntity */
            foreach ($collection as $subEntity) {
                $subEntityId = $subEntity->getId();
                if ($subEntityId === $id) {
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                // Add
                $subEntity = $this->getRoleRepository()->find($id);
                if (!$subEntity) {
                    throw new InvalidArgumentException(sprintf('Mail chimp list with ID %d not found', $id));
                }

                $collection->add($subEntity);
            }
        }

        /** @var MailChimpList $subEntity */
        foreach ($collection as $subEntity) {
            $found = false;
            foreach ($idList as $id) {
                $subEntityId = $subEntity->getId();
                if ($subEntityId === $id) {
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                // Remove
                $collection->removeElement($subEntity);
            }
        }
    }

    public function updateMailChimpGroupName(Collection $collection, array $idList): void
    {
        foreach ($idList as $id) {
            $found = false;
            /** @var MailChimpGroupName $subEntity */
            foreach ($collection as $subEntity) {
                $subEntityId = $subEntity->getId();
                if ($subEntityId === $id) {
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                // Add
                $subEntity = $this->getRoleRepository()->find($id);
                if (!$subEntity) {
                    throw new InvalidArgumentException(sprintf('Mail chimp interest with ID %d not found', $id));
                }

                $collection->add($subEntity);
            }
        }

        /** @var MailChimpGroupName $subEntity */
        foreach ($collection as $subEntity) {
            $found = false;
            foreach ($idList as $id) {
                $subEntityId = $subEntity->getId();
                if ($subEntityId === $id) {
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                // Remove
                $collection->removeElement($subEntity);
            }
        }
    }

    public function updateMailChimpSegment(Collection $collection, array $idList): void
    {
        foreach ($idList as $id) {
            $found = false;
            /** @var MailChimpSegment $subEntity */
            foreach ($collection as $subEntity) {
                $subEntityId = $subEntity->getId();
                if ($subEntityId === $id) {
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                // Add
                $subEntity = $this->getRoleRepository()->find($id);
                if (!$subEntity) {
                    throw new InvalidArgumentException(sprintf('Mail chimp segment with ID %d not found', $id));
                }

                $collection->add($subEntity);
            }
        }

        /** @var MailChimpSegment $subEntity */
        foreach ($collection as $subEntity) {
            $found = false;
            foreach ($idList as $id) {
                $subEntityId = $subEntity->getId();
                if ($subEntityId === $id) {
                    $found = true;
                    break;
                }
            }

            if (!$found) {
                // Remove
                $collection->removeElement($subEntity);
            }
        }
    }
}
