<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\CanadaPost\Model;

use XCart\Extender\Mapping\Extender;
use Doctrine\ORM\Mapping as ORM;
use XC\CanadaPost\Model\Order\Parcel;

/**
 * Class represents an order
 * @Extender\Mixin
 */
abstract class Order extends \XLite\Model\Order
{
    /**
     * Canada Post parcels (reference to the Canada Post parcels model)
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @ORM\OneToMany (targetEntity="XC\CanadaPost\Model\Order\Parcel", mappedBy="order", cascade={"all"})
     */
    protected $capostParcels;

    /**
     * Reference to the Canada Post returns model
     *
     * @var \Doctrine\Common\Collections\Collection
     *
     * @ORM\OneToMany (targetEntity="XC\CanadaPost\Model\ProductsReturn", mappedBy="order", cascade={"all"})
     */
    protected $capostReturns;

    /**
     * Reference to the Canada Post post office model
     *
     * @var \XC\CanadaPost\Model\Order\PostOffice
     *
     * @ORM\OneToOne (targetEntity="XC\CanadaPost\Model\Order\PostOffice", mappedBy="order", cascade={"all"})
     */
    protected $capostOffice;

    // {{{ Service methods

    /**
     * Constructor
     *
     * @param array $data Entity properties OPTIONAL
     */
    public function __construct(array $data = [])
    {
        $this->capostParcels = new \Doctrine\Common\Collections\ArrayCollection();

        parent::__construct($data);
    }

    /**
     * Add Canada Post parcel to order
     *
     * @param \XC\CanadaPost\Model\Order\Parcel $newParcel Canada Post parcel model
     *
     * @return void
     */
    public function addCapostParcel(\XC\CanadaPost\Model\Order\Parcel $newParcel)
    {
        $newParcel->setOrder($this);
        $this->addCapostParcels($newParcel);
    }

    /**
     * Add Canada Post return
     *
     * @param \XC\CanadaPost\Model\ProductsReturn $return Canada Post return model
     *
     * @return void
     */
    public function addCapostReturn(\XC\CanadaPost\Model\ProductsReturn $return)
    {
        $return->setOrder($this);
        $this->addCapostReturns($return);
    }

    /**
     * Set post office
     *
     * @param \XC\CanadaPost\Model\Order\PostOffice $office Post office model (OPTIONAL)
     *
     * @return void
     */
    public function setCapostOffice(\XC\CanadaPost\Model\Order\PostOffice $office = null)
    {
        if ($office !== null) {
            $office->setOrder($this);
        }

        $this->capostOffice = $office;
    }

    // }}}

    /**
     * Return transmitted parcels
     *
     * @return array
     */
    protected function getTransmittedParcels()
    {
        return $this->getCapostParcels()->filter(static function ($capostParcel) {
            return $capostParcel->getStatus() === Parcel::STATUS_TRANSMITED;
        });
    }

    /**
     * Return not transmitted parcels
     *
     * @return array
     */
    protected function getNotTransmittedParcels()
    {
        return $this->getCapostParcels()->filter(static function ($capostParcel) {
            return $capostParcel->getStatus() !== Parcel::STATUS_TRANSMITED;
        });
    }

    /**
     * Remove not transmitted parcels
     *
     * @return void
     */
    protected function removeNotTransmittedCapostParcels()
    {
        $repo = \XLite\Core\Database::getRepo('XC\CanadaPost\Model\Order\Parcel');

        $repo->deleteInBatch($this->getNotTransmittedParcels()->toArray());
    }

    /**
     * Remove irrelevant parcel items
     *
     * @param array $packages
     *
     * @return array
     */
    protected function removeIrrelevantParcelItems()
    {
        foreach ($this->getTransmittedParcels() as $transmittedParcel) {
            $transmittedParcel->removeIrrelevantParcelItems();
        }
    }

    /**
     * Set items
     *
     * @param array $items
     *
     * @return \XLite\Model\OrderItem
     */
    public function setItems($items)
    {
        $this->items = $items;

        return $this;
    }


    /**
     * Remove transmitted items from modifier and return result
     *
     * @param \XLite\Model\Order\Modifier $modifier
     *
     * @return array
     */
    protected function getModifierWithoutTransmittedItems($modifier)
    {
        $transmittedItems = array_reduce(
            $this->getTransmittedParcels()->toArray(),
            static function ($items, $parcel) {
                return array_merge($items, $parcel->getItems()->toArray());
            },
            []
        );

        $items = $modifier->getOrder()->getItems()->toArray();

        foreach ($items as $key => $item) {
            foreach ($transmittedItems as $transmittedItem) {
                if ($transmittedItem->getOrderItem()->getId() === $item->getId()) {
                    $newAmount = $item->getAmount() - $transmittedItem->getAmount();

                    if ($newAmount <= 0) {
                        unset($items[$key]);
                    } else {
                        if (!$item->getCopyOfAmount()) {
                            $item->setCopyOfAmount();
                        }
                        $item->setAmount($newAmount);
                    }
                }
            }
        }

        $modifier->getOrder()->setItems($items);

        return $modifier;
    }

    /**
     * Restore order items counts
     *
     * @return void
     */
    protected function restoreOrderItemsCounts()
    {
        foreach ($this->getItems() as $item) {
            if ($item->getCopyOfAmount()) {
                $item->setAmount($item->getCopyOfAmount());
            }
        }
    }

    /**
     * Return max parcel number
     *
     * @return integer
     */
    protected function getMaxParcelNumber()
    {
        $result = 0;

        foreach ($this->getCapostParcels() as $parcel) {
            $result = max($result, $parcel->getNumber());
        }

        return $result;
    }

    /**
     * Calculated and create Canada Post parcels for the order
     *
     * @return void
     */
    public function createCapostParcels($replace = false)
    {
        $modifier = $this->getModifier(\XLite\Model\Base\Surcharge::TYPE_SHIPPING, 'SHIPPING');

        $capostConfig = \XLite\Core\Config::getInstance()->XC->CanadaPost;

        $maxParcelNumber = 0;
        if ($replace) {
            $this->removeNotTransmittedCapostParcels();
            $this->removeIrrelevantParcelItems();
            $modifier = $this->getModifierWithoutTransmittedItems($modifier);

            if (!$this->getCapostParcels()->isEmpty()) {
                $maxParcelNumber = $this->getMaxParcelNumber();
            }
        }

        $rawPackages = ($modifier && $modifier->getMethod() && $modifier->getMethod()->getProcessorObject())
            ? $modifier->getMethod()->getProcessorObject()->getPackages($modifier)
            : [];

        if ($replace) {
            $this->restoreOrderItemsCounts();
        }

        if (
            !empty($rawPackages)
            && is_array($rawPackages)
        ) {
            foreach ($rawPackages as $packageIdx => $package) {
                if (!$package['items']) {
                    continue;
                }
                $parcel = new \XC\CanadaPost\Model\Order\Parcel();

                $parcel->setOrder($this);

                $parcel->setNumber($packageIdx + 1 + $maxParcelNumber);
                $parcel->setQuoteType($capostConfig->quote_type);

                // Set parcel box dimensions and weight
                foreach (['weight', 'length', 'width', 'height'] as $_param) {
                    $parcel->{'setBox' . \Includes\Utils\Converter::convertToUpperCamelCase($_param)}($package['box'][$_param]);
                }

                // Set parcel types (DEFAULT)
                foreach (['document', 'unpackaged', 'mailing_tube', 'oversized'] as $_param) {
                    $parcel->{'setIs' . \Includes\Utils\Converter::convertToUpperCamelCase($_param)}($capostConfig->{$_param});
                }

                // Set default parcel options (from the module settings)
                $optClasses = $parcel->getAllowedOptionClasses();

                foreach ($optClasses as $optClass => $classData) {
                    if ($parcel::OPT_CLASS_COVERAGE == $optClass) {
                        // Do not use default settings for "coverage"
                        continue;
                    }

                    // Set default option value from module settings
                    $value = (isset($capostConfig->{$optClass}))
                        ? $capostConfig->{$optClass}
                        : '';

                    if (
                        $classData[$parcel::OPT_SCHEMA_MULTIPLE]
                        && $classData[$parcel::OPT_SCHEMA_MANDATORY]
                        && !isset($classData[$parcel::OPT_SCHEMA_ALLOWED_OPTIONS][$value])
                    ) {
                        $opts = array_keys($classData[$parcel::OPT_SCHEMA_ALLOWED_OPTIONS]);
                        // Set allowed option value
                        $value = array_shift($opts);
                    }

                    $parcel->{'setOpt' . \Includes\Utils\Converter::convertToUpperCamelCase($optClass)}($value);
                }

                // Add items to the parcel
                foreach ($package['items'] as $item) {
                    $parcelItem = new \XC\CanadaPost\Model\Order\Parcel\Item();

                    $parcelItem->setAmount($item['qty']);
                    $parcelItem->setOrderItem($this->getItemById($item['id']));

                    // Assign parcel to the order
                    $parcel->addItem($parcelItem);
                }

                // Assign parcel to the order
                $this->addCapostParcel($parcel);

                \XLite\Core\Database::getEM()->persist($parcel);
            }

            \XLite\Core\Database::getEM()->flush();
        }
    }

    /**
     * Get order item by ID
     *
     * @param integer $id Order item ID
     *
     * @return \XLite\Model\OrderItem
     */
    public function getItemById($id)
    {
        foreach ($this->getItems() as $item) {
            if ($item->getItemId() == $id) {
                return $item;
            }
        }

        return null;
    }

    /**
     * Get shipping method code
     *
     * @return string
     */
    public function getCapostShippingMethodCode()
    {
        $modifier = $this->getModifier(\XLite\Model\Base\Surcharge::TYPE_SHIPPING, 'SHIPPING');
        $method = $modifier ? $modifier->getMethod() : null;

        return ($method && $method->getProcessor() === 'capost') ? $method->getCode() : '';
    }

    /**
     * Check - is shipping method is one of the Canada Post or not
     *
     * @return boolean
     */
    public function isCapostShippingMethod()
    {
        $modifier = $this->getModifier(\XLite\Model\Base\Surcharge::TYPE_SHIPPING, 'SHIPPING');
        $method = $modifier ? $modifier->getMethod() : null;

        return $method && $method->getProcessor() === 'capost';
    }

    /**
     * Return Canada Post parcels number
     *
     * @return integer
     */
    public function countCapostParcels()
    {
        return count($this->getCapostParcels());
    }

    /**
     * Check whether order has Canada Post parcels or not
     *
     * @return boolean
     */
    public function hasCapostParcels()
    {
        return (bool) $this->countCapostParcels();
    }

    /**
     * Get Canada Post parcels tracking PINs
     *
     * @return array|null
     */
    public function getCapostTrackingPins()
    {
        $result = [];

        if ($this->hasCapostParcels()) {
            foreach ($this->getCapostParcels() as $parcel) {
                if ($parcel->hasShipment()) {
                    $result[] = [
                        'shipment_id' => $parcel->getShipment()->getId(),
                        'pin_number'  => $parcel->getShipment()->getTrackingPin(),
                    ];
                }
            }
        }

        return (empty($result)) ? null : $result;
    }

    /**
     * Get Canada Post delivery service details
     *
     * @return \XC\CanadaPost\Model\DeliveryService|null
     */
    public function getCapostDeliveryService()
    {
        $service = null;

        if ($this->isCapostShippingMethod()) {
            $serviceCode = $this->getSelectedShippingRate()->getMethod()->getCode();
            $destCountry = ($this->getCapostOffice())
                ? 'CA'
                : $this->getProfile()->getShippingAddress()->getCountry()->getCode();

            $service = \XLite\Core\Database::getRepo('XC\CanadaPost\Model\DeliveryService')
                ->getServiceByCode($serviceCode, $destCountry);
        }

        return $service;
    }

    // {{{ Order fingerprints methods

    /**
     * Define fingerprint keys
     *
     * @return array
     */
    protected function defineFingerprintKeys()
    {
        $list = parent::defineFingerprintKeys();
        $list[] = 'capostOfficeId';
        $list[] = 'capostShippingZipCode';

        // TODO: add 'capostOfficesHash' like for shippings?
        // TODO: add 'deliverToPO' like 'sameAddress' for billing address?

        return $list;
    }

    /**
     * Get fingerprint by 'capostOfficeId' key
     *
     * @return array
     */
    protected function getFingerprintByCapostOfficeId()
    {
        return $this->getCapostOffice()
            ? $this->getCapostOffice()->getOfficeId()
            : 0;
    }

    /**
     * Get fingerprint by 'capostShippingZipCode' field
     *
     * @return string
     */
    protected function getFingerprintByCapostShippingZipCode()
    {
        return ($this->getProfile() && $this->getProfile()->getShippingAddress())
            ? $this->getProfile()->getShippingAddress()->getZipcode()
            : '';
    }

    // }}}

    // {{{ Post Office

    /**
     * Common method to update cart/order
     *
     * @return void
     */
    public function updateOrder()
    {
        parent::updateOrder();

        $this->renewCapostOffice();
    }

    /**
     * Renew selected Canada Post post office
     *
     * @return void
     */
    public function renewCapostOffice()
    {
        $selectedOffice = $this->getCapostOffice();

        if (isset($selectedOffice)) {
            if ($this->isCapostOfficeApplicable($selectedOffice)) {
                $this->setCapostOffice($selectedOffice);
            } else {
                $this->setCapostOffice(null);

                \XLite\Core\Database::getEM()->remove($selectedOffice);
            }
        }
    }

    /**
     * Check - is Canada Post post office is applicable for this order or not
     *
     * @param \XC\CanadaPost\Model\Order\PostOffice $office Canada Post post office object
     *
     * @return boolean
     */
    public function isCapostOfficeApplicable(\XC\CanadaPost\Model\Order\PostOffice $office)
    {
        return $this->isSelectedMethodSupportDeliveryToPO()
            && $this->isCapostOfficeAvailable($office->getOfficeId());
    }

    /**
     * Get selected shipping rate
     *
     * @return \XLite\Model\Shipping\Rate|null
     */
    public function getSelectedShippingRate()
    {
        $modifier = $this->getModifier(\XLite\Model\Base\Surcharge::TYPE_SHIPPING, 'SHIPPING');

        return ($modifier && $modifier->getSelectedRate()) ? $modifier->getSelectedRate() : null;
    }

    /**
     * Get nearest Canada Post post offices that allowed deliver to them list (by the shipping address zipcode)
     *
     * @return array|null
     */
    public function getNearestCapostOffices()
    {
        return $this->getProfile() && $this->getProfile()->getShippingAddress()
            ? \XC\CanadaPost\Core\Service\PostOffice::getInstance()
                ->callGetNearestPostOfficeByZipCode($this->getProfile()->getShippingAddress()->getZipcode(), true)
            : null;
    }

    /**
     * Check - is Canada Post post office is available
     *
     * @param string $officeId Canada Post post office ID
     *
     * @return boolean
     */
    public function isCapostOfficeAvailable($officeId)
    {
        $officesList = $this->getNearestCapostOffices();
        if (isset($officesList) && is_array($officesList)) {
            foreach ($officesList as $v) {
                if ($v->getId() == $officeId) {
                    return true;
                }
            }
        }

        return false;
    }

    /**
     * Check - is selected shipping method support delivery to post office
     *
     * @return boolean
     */
    public function isSelectedMethodSupportDeliveryToPO()
    {
        $rate = $this->getSelectedShippingRate();

        return $rate
            && $rate->getMethod()->getProcessor() === 'capost'
            && in_array(
                $rate->getMethod()->getCode(),
                \XC\CanadaPost\Core\API::getAllowedForDelivetyToPOMethodCodes(),
                true
            );
    }

    // }}}

    /**
     * Add capostParcels
     *
     * @param \XC\CanadaPost\Model\Order\Parcel $capostParcels
     * @return Order
     */
    public function addCapostParcels(\XC\CanadaPost\Model\Order\Parcel $capostParcels)
    {
        $this->capostParcels[] = $capostParcels;
        return $this;
    }

    /**
     * Get capostParcels
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getCapostParcels()
    {
        return $this->capostParcels;
    }

    /**
     * Add capostReturns
     *
     * @param \XC\CanadaPost\Model\ProductsReturn $capostReturns
     * @return Order
     */
    public function addCapostReturns(\XC\CanadaPost\Model\ProductsReturn $capostReturns)
    {
        $this->capostReturns[] = $capostReturns;
        return $this;
    }

    /**
     * Get capostReturns
     *
     * @return \Doctrine\Common\Collections\Collection
     */
    public function getCapostReturns()
    {
        return $this->capostReturns;
    }

    /**
     * Get capostOffice
     *
     * @return \XC\CanadaPost\Model\Order\PostOffice
     */
    public function getCapostOffice()
    {
        return $this->capostOffice;
    }
}
