<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\BuyButton\Core;

/**
 * HTTP-based printer
 */
class Printer
{
    /**
     * Last modified time
     *
     * @var   integer
     */
    protected $lastModified;

    /**
     * Print output
     *
     * @param mixed $data Data
     *
     * @return void
     */
    public function printOutput($data)
    {
        $this->printRaw($data);
    }

    // {{{ Print raw data

    /**
     * Print raw data
     *
     * @param string $data Data
     *
     * @return void
     */
    protected function printRaw($data)
    {
        header('Last-Modified: ' . date('r', $this->getLastModified()));
        header('ETag: ' . $this->getETag($data));

        if (!$this->processAs304($data)) {
            header('Content-Length: ' . strlen($data));
            header('Content-MD5: ' . base64_encode(md5($data, true)));
            print ($data);
        }
    }

    /**
     * Get last modified time
     *
     * @return integer
     */
    protected function getLastModified()
    {
        if (!isset($this->lastModified)) {
            $this->lastModified = $this->defineLastModified();
        }

        return $this->lastModified;
    }

    /**
     * Define last modified time
     *
     * @return integer
     */
    protected function defineLastModified()
    {
        return \XLite\Core\Converter::time();
    }

    /**
     * Get data ETag
     *
     * @param string $data Data
     *
     * @return string
     */
    protected function getETag($data)
    {
        return md5($data);
    }

    /**
     * Process request as 304
     *
     * @param string $data Data
     *
     * @return boolean
     */
    protected function processAs304($data)
    {
        $result = false;

        if (!empty($_SERVER['HTTP_IF_MODIFIED_SINCE'])) {
            $result = strtotime($_SERVER['HTTP_IF_MODIFIED_SINCE']) > $this->getLastModified();
        }

        if (!$result && !empty($_SERVER['HTTP_IF_NONE_MATCH'])) {
            $result = strtolower($_SERVER['HTTP_IF_NONE_MATCH']) == $this->getETag($data);
        }

        if ($result) {
            header('Status: 304 Not Modified', true, 304);
        }

        return $result;
    }

    // }}}
}
