<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\BuyButton\Controller\Features;

use XCart\Domain\ModuleManagerDomain;
use XLite\Core\Config;
use XLite\Core\Request;
use XLite\Core\Router;

/**
 * Provides common buy button helpers
 */
trait BuyButtonProviderTrait
{
    /**
     * @return string
     */
    public function getDefaultIframeWidth()
    {
        return '264';
    }

    /**
     * @return string
     */
    public function getDefaultIframeHeight()
    {
        return '436';
    }

    /**
     * Returns product action url
     *
     * @return string
     */
    public function getActionUrl()
    {
        $action = Request::getInstance()->click_action ?? 'details';

        switch ($action) {
            case 'add':
                return $this->getAddToCartUrl();
                 break;

            case 'details':
            default:
                return $this->getProductUrl();
                 break;
        }
    }

    /**
     * Returns request product card language
     */
    public function getCardLanguage()
    {
        $lang = Request::getInstance()->language ?? 'session';

        if ($lang === 'session') {
            return null;
        }

        if ($lang === 'default') {
            $lang = Config::getInstance()->General->default_language;
        }

        $translation = \XLite\Core\Database::getRepo('\XLite\Model\Language')->findOneBy([
            'code' => $lang,
            'added' => true,
            'enabled' => true
        ]);

        return $translation
            ? $lang
            : null;
    }

    /**
     * Returns request product currency if multicurrency module is active
     */
    public function getCardCurrency()
    {
        /** @var ModuleManagerDomain $moduleManagerDomain */
        $moduleManagerDomain = \XCart\Container::getContainer()?->get(ModuleManagerDomain::class);

        if ($moduleManagerDomain->isEnabled('XC-MultiCurrency')) {
            $core = \XC\MultiCurrency\Core\MultiCurrency::getInstance();
            $currencyCode = Request::getInstance()->currency ?? 'default';

            $currency = null;

            if ($currencyCode === 'default') {
                $currency = $core->getDefaultCurrency();
            } else {
                /** @var \XC\MultiCurrency\Model\ActiveCurrency $active */
                $active = \XLite\Core\Database::getRepo('XC\MultiCurrency\Model\ActiveCurrency')->getCurrencyByCode($currencyCode);

                if ($active) {
                    $currency = $active->getCurrency();
                }
            }

            return $currency;
        }

        return null;
    }

    /**
     * Returns additional params for button part of the buy button
     */
    protected function getAdditionalUrlParams()
    {
        $params = [];

        if ($this->getCardLanguage()) {
            $params['force_lang'] = $this->getCardLanguage();
        }

        if ($this->getCardCurrency()) {
            $params['currency'] = $this->getCardCurrency()->getCode();
        }

        return $params;
    }

    /**
     * Return add to cart action url
     *
     * @return string
     */
    public function getAddToCartUrl()
    {
        $params = array_merge([
            'product_id' => Request::getInstance()->product_id ?: 0,
            'attribute_values' => Request::getInstance()->attribute_values,
            'bb_return' => true,
            'returnURL' => $this->getCartUrl(),
        ], $this->getAdditionalUrlParams());

        Router::getInstance()->disableLanguageUrlsTmp();

        $url = \XLite\Core\Converter::buildFullURL(
            'cart',
            'add',
            $params,
            \XLite::getInstance()->getCustomerScript()
        );

        Router::getInstance()->releaseLanguageUrlsTmp();

        return $url;
    }

    /**
     * Returns product details url
     *
     * @return string
     */
    public function getProductUrl()
    {
        $params = array_merge([
            'product_id' => Request::getInstance()->product_id ?: 0,
            'attribute_values' => Request::getInstance()->attribute_values,
        ], $this->getAdditionalUrlParams());

        Router::getInstance()->disableLanguageUrlsTmp();

        $url = \XLite\Core\Converter::buildFullURL(
            'product',
            '',
            $params,
            \XLite::getInstance()->getCustomerScript()
        );

        Router::getInstance()->releaseLanguageUrlsTmp();

        return $url;
    }

    /**
     * Returns product details url
     *
     * @return string
     */
    public function getCartUrl()
    {
        Router::getInstance()->disableLanguageUrlsTmp();

        $url = \XLite\Core\Converter::buildFullURL(
            'cart',
            '',
            [],
            \XLite::getInstance()->getCustomerScript()
        );

        Router::getInstance()->releaseLanguageUrlsTmp();

        return $url;
    }
}
