<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\BulkEditing\Logic\RemoveProducts\Step;

use ArrayObject;
use Doctrine\ORM\Internal\Hydration\HydrationException;
use Exception;
use Iterator;
use XLite\Core\Database;
use XLite\InjectLoggerTrait;
use XLite\Logic\ARepoStep;
use XLite\Model\AEntity;
use XLite\Model\Product;

class Products extends ARepoStep
{
    use InjectLoggerTrait;

    // {{{ SeekableIterator, Countable

    public function count(): int
    {
        if (!isset($this->countCache)) {
            $optionsIndex = 'count' . get_class($this);
            $options = $this->generator->getOptions();

            if (!isset($options[$optionsIndex])) {
                $options[$optionsIndex] = $this->generator::getTotalProductsCount();
            }

            $this->countCache = $options[$optionsIndex];
        }

        return $this->countCache;
    }

    // }}}

    // {{{ Row processing

    public function run()
    {
        $time = microtime(true);

        $this->generator->setInProgress(true);

        $this->removeProduct($this->getItems()->current());

        $this->generator->setInProgress(false);

        $this->generator->getOptions()->time += round(microtime(true) - $time, 3);

        return true;
    }

    protected function removeProduct(int $productId): void
    {
        // deleteById throws an exception if there is no product with ID = $productId in the database.
        try {
            Database::getRepo(Product::class)?->deleteById($productId, false);
        } catch (Exception) {
        }
    }

    // }}}

    // {{{ Data

    /**
     * @throws HydrationException
     */
    protected function getItems($reset = false): Iterator
    {
        if (!isset($this->items) || $reset) {
            $productIds = new ArrayObject($this->generator::getProducts());
            $this->items = $productIds->getIterator();
            $this->items->rewind();
        }

        return $this->items;
    }

    // }}}

    // {{{ Data

    protected function getRepository(): ?\XLite\Model\Repo\Product
    {
        return Database::getRepo(Product::class);
    }

    protected function processModel(AEntity $model): void
    {
        Database::getEM()->remove($model);
    }

    // }}}
}
