<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\BulkEditing\Logic\RemoveProducts;

use XLite\Core\CommonCell;
use XLite\Core\Database;
use XC\BulkEditing\Core\EventListener\RemoveProducts;
use XLite\Core\EventTask;
use XLite\Core\Session;
use XLite\Logic\AGenerator;
use XC\BulkEditing\Logic\RemoveProducts\Step\Products;
use XLite\Model\Product as ProductModel;
use XLite\Model\Repo\ARepo;
use XLite\Model\TmpVar;
use XLite\View\ItemsList\Model\Product\Admin\AAdmin;

class Generator extends AGenerator
{
    protected static bool $inProgress = false;

    protected static AAdmin|null|bool $listWidget = false;

    protected static CommonCell|null|bool $productSearchCnd = false;

    public function setInProgress($value): void
    {
        static::$inProgress = $value;
    }

    // {{{ Steps

    /**
     * @return string[]
     */
    protected function getStepsList(): array
    {
        return [
            Products::class
        ];
    }

    /**
     * @return string[]
     */
    protected function defineSteps(): array
    {
        return $this->getStepsList();
    }

    // }}}

    // {{{ SeekableIterator, Countable

    /**
     * @return int
     */
    public function count(): int
    {
        if (!isset($this->countCache)) {
            if (!isset($this->options['count'])) {
                $this->options['count'] = 0;

                foreach ($this->getSteps() as $step) {
                    $this->options['count'] += $step->count();
                    $this->options['count' . get_class($step)] = $step->count();
                }
            }

            $this->countCache = $this->options['count'];
        }

        return $this->countCache;
    }

    // }}}

    // {{{ Service variable names

    public static function getEventName(): string
    {
        return 'removeProducts';
    }

    public static function getTotalProductsCountVarName(): string
    {
        return static::getEventName() . 'TotalProducts';
    }

    public static function getLastEventTotalProductsCount(): int
    {
        return (int)(Database::getRepo(TmpVar::class)?->getVar(static::getTotalProductsCountVarName()));
    }

    public static function removeLastEventTotalProductsCount(): void
    {
        Database::getRepo(TmpVar::class)?->removeVar(static::getTotalProductsCountVarName());
    }

    /**
     * @return int[]|int Total products count to delete if $countOnly is true, otherwise an array of product ids to
     *                   remove on the current step.
     */
    public static function getProducts(bool $countOnly = false, ?string $listWidgetFQN = null): array|int
    {
        $result = $countOnly ? 0 : [];
        $cndCell = static::getConditionCell($listWidgetFQN);

        if ($cndCell) {
            $result = Database::getRepo(ProductModel::class)?->search(
                $cndCell,
                ($countOnly ? ARepo::SEARCH_MODE_COUNT : ARepo::SEARCH_MODE_IDS)
            );
        }

        return $result;
    }

    public static function getEventState(): array
    {
        return (array)(
            Database::getRepo(TmpVar::class)?->getEventState(
                static::getEventName()
            ) ?? []
        );
    }

    public static function isCancelled(): bool
    {
        return (bool)Database::getRepo(TmpVar::class)?->getVar(static::getCancelFlagVarName());
    }

    public static function isInProgress(string $listWidgetFQN): bool
    {
        $eventState = static::getEventState();

        return $eventState
               && (string)($eventState['options']['widget'] ?? '') === $listWidgetFQN
               && in_array(
                   (int)($eventState['state'] ?? 0),
                   [
                       EventTask::STATE_STANDBY,
                       EventTask::STATE_IN_PROGRESS
                   ],
                   true
               )
               && !static::isCancelled();
    }

    // }}}

    protected static function getListWidgetFQN(): string
    {
        return (string)(static::getEventState()['options']['widget'] ?? '');
    }

    protected static function getListWidget(?string $listWidgetFQN = null): ?AAdmin
    {
        $itemsListWidgetFQN = $listWidgetFQN ?? static::getListWidgetFQN();

        if (static::$listWidget === false) {
            static::$listWidget = class_exists($itemsListWidgetFQN) ? new $itemsListWidgetFQN() : null;
        }

        return static::$listWidget;
    }

    protected static function getConditionCell(?string $listWidgetFQN = null): ?CommonCell
    {
        if (static::$productSearchCnd === false) {
            $eventState = static::getEventState();

            if (
                !empty($eventState['options']['productIds'])
                && is_array($eventState['options']['productIds'])
                && !in_array(
                    (int)($eventState['state'] ?? 0),
                    [EventTask::STATE_FINISHED, EventTask::STATE_ABORTED],
                    true
                )
            ) {
                static::$productSearchCnd = new CommonCell();
                static::$productSearchCnd->ids = $eventState['options']['productIds'];
            } else {
                static::$productSearchCnd = (
                    ($listWidget = static::getListWidget($listWidgetFQN))
                    && method_exists($listWidget, 'getConditionCellName')
                ) ? Session::getInstance()->{$listWidget::getConditionCellName()} : null;
            }

            if (static::$productSearchCnd) {
                static::$productSearchCnd->limit = [0, RemoveProducts::CHUNK_LENGTH];
                static::$productSearchCnd->orderBy = ['p.product_id', 'ASC'];
            }
        }

        return static::$productSearchCnd;
    }

    public static function getTotalProductsCount(?string $listWidgetFQN = null): int
    {
        return static::getProducts(true, $listWidgetFQN);
    }

    public static function run(array $options): void
    {
        parent::run($options);

        Database::getRepo(TmpVar::class)?->setVar(
            static::getTotalProductsCountVarName(),
            static::getTotalProductsCount()
        );
    }
}
