<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\BulkEditing\Core\EventListener;

use XC\BulkEditing\Logic\RemoveProducts\Generator;
use XLite\Core\Database;
use XCart\Container;
use XLite\Core\Translation;
use XLite\Model\TmpVar;

class RemoveProducts extends \XLite\Core\EventListener\Base\Countable
{
    public const CHUNK_LENGTH = 100;

    /** @var Generator $generator */
    protected $generator;

    protected int $timeMark = 0;

    protected int $serviceTime = 0;

    protected int $counter = self::CHUNK_LENGTH;

    protected function getEventName(): string
    {
        return 'removeProducts';
    }

    protected function processItem($item): bool
    {
        $this->serviceTime += (microtime(true) - $this->timeMark);

        $result = $item->run();

        $this->timeMark = microtime(true);

        if (!$this->getItems()->valid()) {
            $result = false;
            foreach ($this->getItems()->getErrors() as $error) {
                $this->errors[] = $error['title'];
            }
        }

        return $result;
    }

    protected function isStepValid(): bool
    {
        return parent::isStepValid()
            && $this->getItems()->valid();
    }

    protected function getLength(): int
    {
        return $this->getItems()->count();
    }

    protected function getItems(): Generator
    {
        if (!isset($this->generator)) {
            $this->generator = new Generator(
                $this->record['options'] ?? []
            );
        }

        return $this->generator;
    }

    protected function initializeStep(): void
    {
        $this->timeMark = microtime(true);

        set_time_limit(0);
        $this->counter = static::CHUNK_LENGTH;

        Container::getServiceLocator()->getCallbackManager()->setContextValue('remove-products');

        parent::initializeStep();
    }

    protected function finishStep(): void
    {
        $generator = $this->getItems();

        $this->serviceTime += (microtime(true) - $this->timeMark);
        $generator->getOptions()->time += $this->serviceTime;

        $this->record['options'] = $generator->getOptions()->getArrayCopy();
        Database::getEM()->flush();

        parent::finishStep();
    }

    protected function finishTask(): void
    {
        $this->record['options'] = $this->getItems()->getOptions()->getArrayCopy();
        Database::getEM()->flush();

        parent::finishTask();

        $this->getItems()->finalize();
    }

    protected function compileTouchData(): void
    {
        $timeLabel = Translation::formatTimePeriod($this->getItems()->getTimeRemain());
        $this->record['touchData'] = [];
        if ($timeLabel) {
            $this->record['touchData']['message'] = (string)static::t('About X remaining', ['time' => $timeLabel]);
        }
    }

    protected function isStepSuccess(): bool
    {
        return parent::isStepSuccess() && !$this->getItems()->hasErrors();
    }

    protected function isContinue($item): bool
    {
        $this->counter--;

        return parent::isContinue($item) && 0 < $this->counter && empty($this->errors);
    }

    protected function failTask(): void
    {
        parent::failTask();

        Database::getRepo(TmpVar::class)?->removeEventState($this->getEventName());
    }
}
