<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace XC\BulkEditing\Controller\Admin;

use XLite\Controller\Admin\AAdmin;
use XLite\Core\Request;
use XLite\Core\Converter;
use XLite\Core\TopMessage;
use XC\BulkEditing\Logic\RemoveProducts\Generator;
use XC\BulkEditing\View\Popup\RemoveProducts as RemoveProductsPopup;
use XLite\Core\Translation\Label;

class BulkRemoveProducts extends AAdmin
{
    protected ?Generator $removeProductsGenerator = null;

    public function getTitle(): string|Label
    {
        return static::t('Delete Products');
    }

    public function isProductDeletionInProgress(): bool
    {
        return Generator::isInProgress((string)Request::getInstance()->listWidget);
    }

    /**
     * @return string[]
     */
    public static function defineFreeFormIdActions(): array
    {
        $actions = parent::defineFreeFormIdActions();
        $actions[] = 'start';

        return $actions;
    }

    protected function doActionStart(): void
    {
        $request = Request::getInstance();
        $listWidget = $request->listWidget;

        if (!$this->isProductDeletionInProgress()) {
            if ($listWidget) {
                $options = [
                    'widget' => $listWidget
                ];

                $productIds = $request->select;
                if (!empty($productIds) && is_array($productIds)) {
                    $options['productIds'] = array_keys($productIds);
                }

                Generator::run($options);
            } else {
                TopMessage::addError('List widget is not set');
            }
        }

        $this->setInternalRedirect();
        $this->returnURL = Converter::buildURL(
            'bulk_remove_products',
            '',
            [
                'widget'     => RemoveProductsPopup::class,
                'listWidget' => $listWidget,
                'returnUrl'  => $this->getReturnURL()
            ]
        );
    }

    public function getRemoveProductsGenerator(): Generator|bool
    {
        if (!isset($this->removeProductsGenerator)) {
            $state = Generator::getEventState();
            $this->removeProductsGenerator = ($state && isset($state['options']))
                ? new Generator($state['options'])
                : false;
        }

        return $this->removeProductsGenerator;
    }

    protected function doActionRemoveProductsCancel(): void
    {
        $eventState = Generator::getEventState();
        $totalProducts = Generator::getLastEventTotalProductsCount();
        Generator::removeLastEventTotalProductsCount();
        Generator::cancel();

        $productsRemoved = (int)($eventState['position'] ?? 0);

        if ($productsRemoved < $totalProducts) {
            TopMessage::addWarning(
                static::t(
                    'Products deletion cancelled. X products out of Y have been deleted',
                    [
                        'removed_products' => $productsRemoved,
                        'total_products'   => $totalProducts
                    ]
                )
            );
        } else {
            TopMessage::addInfo(
                'X products have been deleted.',
                ['products' => $productsRemoved]
            );
        }

        $this->setHardRedirect();
    }

    protected function doNoAction(): void
    {
        $request = Request::getInstance();

        if ($request->remove_completed) {
            TopMessage::addInfo(
                'X products have been deleted.',
                ['products' => Generator::getLastEventTotalProductsCount()]
            );

            Generator::removeLastEventTotalProductsCount();

            $this->setReturnURL(
                $request->returnURL ?? $this->buildURL('bulk_remove_products')
            );
        } elseif ($request->remove_failed) {
            TopMessage::addError('Products deletion has been stopped because of errors in the process.');

            Generator::removeLastEventTotalProductsCount();

            $this->setReturnURL(
                $request->returnURL ?? $this->buildURL('bulk_remove_products')
            );
        }
        parent::doNoAction();
    }
}
