<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\XLSExportImport\Core;

/**
 * XLS file object
 */
class XLSFileObject extends \SplFileObject implements \Countable
{
    /**
     * Position
     *
     * @var integer
     */
    protected $position = 0;

    /**
     * Total number of rows
     *
     * @var integer
     */
    protected $count;

    /**
     * Total number of columns
     *
     * @var integer
     */
    protected $width;

    /**
     * Reader
     *
     * @var \PhpOffice\PhpSpreadsheet\Spreadsheet
     */
    protected $reader;

    /**
     * Open
     */
    public function open(): void
    {
        /** @var \PhpOffice\PhpSpreadsheet\Reader\IReader $reader */
        $reader = \PhpOffice\PhpSpreadsheet\IOFactory::createReaderForFile($this->getPathname());
        $reader->setReadDataOnly(true);

        $this->reader = $reader->load($this->getPathname());
        $sheet = $this->reader->setActiveSheetIndex(0);

        // Count
        $this->count = $sheet->getHighestRow();

        // Width
        $rowInternal = $sheet->getRowIterator(1)->current();
        $cellIterator = $rowInternal->getCellIterator();
        $cellIterator->setIterateOnlyExistingCells(true);

        $this->width = iterator_count($cellIterator);
    }

    /**
     * @return array|false|string
     */
    #[\ReturnTypeWillChange]
    public function current()
    {
        $rowInternal = $this->reader->getActiveSheet()->getRowIterator($this->position + 1)->current();
        $cellIterator = $rowInternal->getCellIterator();
        $cellIterator->setIterateOnlyExistingCells(false);

        $row = [];
        foreach ($cellIterator as $cell) {
            /** @var \PhpOffice\PhpSpreadsheet\Cell\Cell $cell */
            $row[] = $cell->getValue() ?? '';
        }

        // Normalize row's width
        if (count($row) > $this->width) {
            $row = array_slice($row, 0, $this->width);
        } elseif (count($row) < $this->width) {
            for ($i = count($row); $i < $this->width; $i++) {
                $row[] = '';
            }
        }

        return $row;
    }

    /**
     * @return void
     */
    public function rewind(): void
    {
        $this->position = 0;
    }

    /**
     * @return void
     */
    public function next(): void
    {
        do {
            $this->position++;
        } while ($this->valid() && $this->isEmptyRow($this->current()));
    }

    /**
     * @return bool
     */
    public function valid(): bool
    {
        return $this->count > $this->position;
    }

    /**
     * @return mixed
     */
    public function key(): mixed
    {
        return $this->position;
    }

    /**
     * @param int $position
     *
     * @return void
     */
    public function seek(int $position): void
    {
        $this->position = $position;
    }

    /**
     * @return int
     */
    public function count(): int
    {
        return $this->count;
    }

    /**
     * @return bool
     */
    public function eof(): bool
    {
        return !$this->valid();
    }

    /**
     * Check - row is empty or not
     *
     * @param string[] $row Data row
     *
     * @return boolean
     */
    protected function isEmptyRow(array $row)
    {
        $row = array_filter(
            $row,
            static function ($elm) {
                return trim($elm) !== '';
            }
        );

        return count($row) === 0;
    }
}
