<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace QSL\ShopByBrand\Model\DTO\Product;

use Exception;
use QSL\ShopByBrand\Main;
use QSL\ShopByBrand\Model\Brand;
use XLite\Model\AttributeValue\AttributeValueSelect;
use XCart\Extender\Mapping\Extender;
use XLite\Core\Database;
use XLite\Model\Product;
use XLite\Model\Attribute;

/**
 * @Extender\Mixin
 */
class Info extends \XLite\Model\DTO\Product\Info
{
    protected function getProductBrand(Product $product): ?int
    {
        return (
            $product->getProductId()
            && ($brand = Database::getRepo(Brand::class)?->findProductBrand($product))
        ) ? $brand->getBrandId() : null;
    }

    /**
     * @param Product $object
     *
     * @return void
     */
    protected function init($object)
    {
        parent::init($object);

        if (Main::isBrandAttributeSet()) {
            $this->default->brand = $this->getProductBrand($object);
        }
    }

    /**
     * @param Product $object
     * @param array   $rawData
     *
     * @return void
     * @throws Exception
     */
    public function populateTo($object, $rawData = null)
    {
        parent::populateTo($object, $rawData);

        if (Main::isBrandAttributeSet()) {
            $newBrandId = (int) ($rawData['default']['brand'] ?? 0);

            if (
                ($this->getProductBrand($object) !== $newBrandId)
                /** @var Attribute|null $brandAttribute */
                && ($brandAttribute = Database::getRepo(Attribute::class)?->findBrandAttribute())
            ) {
                $attributeValues = [];

                if ($object->getProductId()) {
                    /** @var AttributeValueSelect[] $attributeValues */
                    $attributeValues = Database::getRepo(AttributeValueSelect::class)?->findProductAttributeValues(
                        $object,
                        $brandAttribute
                    ) ?? [];
                }

                /** @var Brand|null $newBrand */
                $newBrand = Database::getRepo(Brand::class)?->findOneBy([
                    'brand_id' => $newBrandId
                ]);

                $entityManager = Database::getEM();

                if ($attributeValues) {
                    foreach ($attributeValues as $i => $value) {
                        if ($i > 0 || !$newBrand) {
                            $entityManager->remove($value);
                        } else {
                            $value->setAttributeOption($newBrand->getOption());
                        }
                    }
                } elseif ($newBrand) {
                    $value = new AttributeValueSelect();
                    $value->setAttributeOption($newBrand->getOption());
                    $value->setAttribute($brandAttribute);
                    $value->setProduct($object);

                    $entityManager->persist($value);
                }
            }
        }
    }
}
