<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

declare(strict_types=1);

namespace QSL\ShopByBrand\Logic\Import\Processor;

use Exception;
use QSL\ShopByBrand\Model\Brand;
use QSL\ShopByBrand\Model\Image\Brand\Image as BrandImage;
use QSL\ShopByBrand\Model\Repo\Brand as BrandRepository;
use XLite\Core\Database;
use XLite\Core\Validator\String\CleanURL as CleanURLValidator;
use XLite\Model\Attribute;
use XLite\Model\AttributeOption;

class Brands extends \XLite\Logic\Import\Processor\AProcessor
{
    protected function detectModel(array $data): ?Brand
    {
        $attributeOption = Database::getRepo(AttributeOption::class)?->getAttributeOptionBrandByName($data['name']);

        return $attributeOption ? $this->getRepository()->findOneBy(['option' => $attributeOption->getId()]) : null;
    }

    public static function getTitle(): string
    {
        return (string) static::t('Brand imported');
    }

    protected function getRepository(): BrandRepository
    {
        return Database::getRepo(Brand::class);
    }

    protected function defineColumns(): array
    {
        return [
            'name'            => [
                static::COLUMN_IS_KEY => true,
                static::COLUMN_LENGTH => 255,
            ],
            'image'           => [],
            'cleanURL'        => [],
            'position'        => [],
            'description'     => [
                static::COLUMN_IS_MULTILINGUAL => true,
                static::COLUMN_IS_TAGS_ALLOWED => true
            ],
            'metaKeywords'    => [
                static::COLUMN_IS_MULTILINGUAL => true,
                static::COLUMN_LENGTH => 255,
            ],
            'metaDescription' => [
                static::COLUMN_IS_MULTILINGUAL => true,
            ],
            'metaTitle'       => [
                static::COLUMN_IS_MULTILINGUAL => true,
                static::COLUMN_LENGTH => 255,
            ],
        ];
    }

    /**
     * @return array
     */
    public static function getMessages()
    {
        return parent::getMessages() +
               [
                   'BRAND-NAME-FMT'             => 'The name is empty',
                   'BRAND-POSITION-FMT'         => 'Wrong position format',
                   'BRAND-IMG-LOAD-FAILED'      => 'Error of image loading. Make sure the "images" directory has write permissions.',
                   'BRAND-IMG-URL-LOAD-FAILED'  => 'Couldn\'t download the image {{value}} from URL',
                   'BRAND-IMG-FILE-LOAD-FAILED' => 'Failed to load the file {{value}} because it does not exist',
               ];
    }

    /**
     * @throws Exception
     */
    protected function verifyName(mixed $value, array $column): void
    {
        $value = $this->getDefLangValue($value);
        if ($this->verifyValueAsEmpty($value)) {
            $this->addError('BRAND-NAME-FMT', ['column' => $column, 'value' => $value]);
        }
    }

    protected function verifyImage(mixed $value, array $column): void
    {
        if (!$this->verifyValueAsEmpty($value) && !$this->verifyValueAsNull($value) && !$this->verifyValueAsFile($value)) {
            $this->addWarning('GLOBAL-IMAGE-FMT', ['column' => $column, 'value' => $value]);
        }
    }

    protected function verifyPosition(mixed $value, array $column): void
    {
        if (!$this->verifyValueAsEmpty($value) && !$this->verifyValueAsFloat($value)) {
            $this->addWarning('BRAND-POSITION-FMT', ['column' => $column, 'value' => $value]);
        }
    }

    protected function importNameColumn(Brand $model, string $value, array $column): void
    {
        $attributeOption = Database::getRepo(AttributeOption::class)?->getAttributeOptionBrandByName($value);

        if (!$attributeOption) {
            $attributeOption = new AttributeOption();
            $attributeOption->setAttribute(
                Database::getRepo(Attribute::class)?->findBrandAttribute()
            );
            $attributeOption->setName($value);
            Database::getEM()->persist($attributeOption);
        }

        $model->setOption($attributeOption);
    }

    protected function importImageColumn(Brand $model, string $value, array $column): void
    {
        if (!$value) {
            return;
        }

        if ($this->verifyValueAsNull($value)) {
            if ($model->getImage()) {
                Database::getEM()->remove($model->getImage());
                $model->setImage();
                Database::getEM()->flush();
            }
        } else {
            $image = $model->getImage() ?: new BrandImage();

            if ($image->loadFromPath($value)) {
                Database::getEM()->persist($image);

                $image->setNeedProcess(true)->setBrand($model);
                $model->setImage($image);
            } else {
                $file = $this->verifyValueAsLocalURL($value) ? $this->getLocalPathFromURL($value) : $value;

                if ($image->getLoadError() === 'unwriteable') {
                    $this->addError('BRAND-IMG-LOAD-FAILED', [
                        'column' => $column,
                        'value'  => $this->verifyValueAsURL($file) ? $value : LC_DIR_ROOT . $file
                    ]);
                } elseif ($image->getLoadError() === 'undownloadable') {
                    $this->addWarning('BRAND-IMG-URL-LOAD-FAILED', [
                        'column' => $column,
                        'value'  => $this->verifyValueAsURL($file) ? $value : LC_DIR_ROOT . $file
                    ]);
                } elseif ($image->getLoadError() === 'nonexistent') {
                    $this->addWarning('BRAND-IMG-FILE-LOAD-FAILED', [
                        'column' => $column,
                        'value'  => $this->verifyValueAsURL($file) ? $value : LC_DIR_ROOT . $file
                    ]);
                } elseif ($image->getLoadError()) {
                    $this->addWarning('BRAND-IMG-URL-LOAD-FAILED', [
                        'column' => $column,
                        'value'  => $this->verifyValueAsURL($file) ? $value : LC_DIR_ROOT . $file
                    ]);
                }
            }
        }
    }

    /**
     * @throws Exception
     */
    protected function importCleanURLColumn(Brand $model, string $value, array $column): void
    {
        if (!\XLite\Core\Converter::isEmptyString($value)) {
            $validator = new CleanURLValidator(false, null, get_class($model), $model->getId());

            try {
                $validator->validate($value);
                $model->setCleanURL($value);
            } catch (\XLite\Core\Validator\Exception $exception) {
            }
        }
    }
}
