/* vim: set ts=2 sw=2 sts=2 et: */

/**
 * Controller
 *
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

var stickerController = {
  config: xcart.getCommentedData(jQuery('body'), 'stickers'),
  topStickerQuantity: 2,
  bottomStickerQuantity: 4,
  productStickerInit: function(base) {
    let self = this;
    let products = base.hasClass('box-product') ? base : base.find('.box-product, .product-cell');

    products.each(function(){
      let product = $(this);

      if (
        product.closest('.product-details').length
        && !self.config.show_stickers_on_product_pages
      ) {
        return;
      }

      let displayMode = self.config.sticker_display_mode;
      let container = $(".product-stickers", product);

      self.buildProductStickers(product);

      if (displayMode === 'corner_ribbon') {
        container.addClass(displayMode + '_mode');
        self.renderTopRibbons(product);
        self.renderBottomRibbons(product);
        self.postRender(product);
      }

      container.toggleClass('show');
    });
  },
  buildProductStickers: function(product) {
    $(".top-stickers ul", product).append(this.getTopStickers(product));
    $(".bottom-stickers ul", product).append(this.getBottomStickers(product));
  },
  renderTopRibbons: function(product){
    let container = $(".top-stickers ul", product);
    let x = container.width();
    let x1 = container.parent().width();
    let y = container.height();
    let y1 = container.parent().height();
    container.css({'left': this.calculateLeftOffset(x, x1), 'top': this.calculateTopOffset(y, y1)});
    container.parent().css({'height': this.calculateContainerHeight(y, y1), 'width': this.calculateContainerWidth(y, x1)});
  },
  postRender: function(product){
    let container = $(".bottom-stickers ul", product);
    $(container.find('li').get().reverse()).each(function(){
      let textContainer = $(this).find(".text-ribbon > span");
      let productItemOffset = container.closest('.product, .product-details').offset();
      let stickerOffset = textContainer.offset();
      let stickerFullWidth = parseInt($(this).css('width'));
      let stickerInitialHeight = parseInt($(this).css('height'));
      let textContainerHeight = parseInt(textContainer.css('height'));
      let stickerPadding = stickerInitialHeight - textContainerHeight;
      let stickerLeftOffset = (productItemOffset.left - stickerOffset.left) * (Math.sqrt(2));
      let stickerCroppedWidth = stickerFullWidth - stickerLeftOffset * 2;

      textContainer.css({
        'max-width': stickerCroppedWidth + 'px',
        'white-space': 'normal'
      });

      let multiplicity = (parseInt($(this).css('height')) - stickerPadding) / (stickerInitialHeight - stickerPadding);
      if (multiplicity > 1) {
        multiplicity--;
        let hypotenuse = multiplicity * textContainerHeight;
        let leg = hypotenuse / (Math.sqrt(2));
        let padding = stickerPadding / 2;
        container.css({
          'left': parseInt(container.css('left')) - hypotenuse - padding,
          'top': parseInt(container.css('top')) + hypotenuse / 2 + padding
        });
        container.parent().css({
          'width': parseInt(container.parent().css('width')) + leg * 2 + padding,
          'height': parseInt(container.parent().css('height')) + leg * 2 + padding
        });
      }
    });
  },
  renderBottomRibbons: function(product){
    let container = $(".bottom-stickers ul", product);
    let x = container.width();
    let x1 = container.parent().width();
    let y = container.height();
    let y1 = container.parent().height();
    container.css({'left': this.calculateLeftOffset(x, x1)-(this.calculateContainerWidth(y, x1)-x1), 'top': this.calculateTopOffset(y, y1)+(this.calculateContainerHeight(y, y1)-y1)});
    container.parent().css({'height': this.calculateContainerHeight(y, y1), 'width': this.calculateContainerWidth(y, x1)});
  },
  getDefaultStickerBase: function(product) {
    const elem = product.get(0);

    const findBox = !!elem.querySelector('.image')
      ? product
      : elem.querySelectorAll('.product-photo');

    return elem
    && !!findBox[0].querySelector('ul.labels')
      ? findBox[0].querySelector('ul.labels')
      : null;
  },
  getCustomStickerBase: function(product) {
    return product ? product.find('.product-stickers > ul') : null;
  },
  getDefaultStickers: function(product) {
    let self = this;
    let base = this.getDefaultStickerBase(product);
    let labels = (base ? $(base).find('li').clone().toArray() : []).concat(
      product.find('.product-freeship-label:not(.product-sticker)').clone().addClass('product-sticker').toArray()
    );

    if (this.config.move_labels) {
      labels = _.each(labels, function (label) {
        _.each(self.config.labels, function (relation) {
          if (
            $(label).hasClass(relation.name)
            || $(label).hasClass('product-freeship-label') && relation.name === 'free-shipping'
          ) {
            $(label).find(".content").css({
              'background': '#' + relation.bg_color,
              'color': '#' + relation.text_color
            });
            return $(label);
          }
        });
      });
    }

    return labels;
  },
  getCustomStickers: function(product) {
    let base = this.getCustomStickerBase(product);
    return base ? base.find('li').clone().toArray() : [];
  },
  getAllStickers: function(product) {
    return this.showLabelsAsStickers()
      ? this.getDefaultStickers(product).concat(this.getCustomStickers(product))
      : this.getCustomStickers(product);
  },
  getTopStickers: function(product) {
    return this.getAllStickers(product).splice(0, this.topStickerQuantity);
  },
  getBottomStickers: function(product) {
    return this.getAllStickers(product).splice(this.topStickerQuantity, this.bottomStickerQuantity);
  },
  calculateLeftOffset: function(x, x1) {
    return ((x1-x)/2);
  },
  calculateTopOffset: function(y, y1) {
    return ((y1-y)/2);
  },
  calculateContainerHeight: function(h, h1) {
    return (h1 + h/(Math.sqrt(2)));
  },
  calculateContainerWidth: function(h, w1) {
    return (w1 + h/(Math.sqrt(2)));
  },
  isProductPage: function() {
    return $("body").hasClass("target-product");
  },
  showLabelsAsStickers: function() {
    return this.config.move_labels;
  },
  isInitAllowed: function() {
    return (
      !this.isProductPage()
      || this.config.show_stickers_on_product_pages
      || (
        this.showLabelsAsStickers()
        && document.getElementsByClassName('items-list-products')?.length
      )
    );
  },
}

xcart.bind(['list.products.postprocess', 'block.product.details.postprocess'], function(event, data) {
  if (stickerController.isInitAllowed()) {
    stickerController.productStickerInit(data.widget.base);
    $(".box-product .infoBlock .labels, .product-cell .product-photo .labels").show();
  }
  if (stickerController.showLabelsAsStickers()) {
    $(".box-product .infoBlock .labels, .product-cell .product-photo .labels").hide();
  }
});
