<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\CostPrice\Model;

use XCart\Extender\Mapping\Extender;
use Doctrine\ORM\Mapping as ORM;

/**
 * @Extender\Mixin
 */
abstract class OrderItem extends \XLite\Model\OrderItem
{
    /**
     * Item cost price
     *
     * @var float
     *
     * @ORM\Column (type="decimal", nullable=true, precision=14, scale=4, options={"default" : 0})
     */
    protected $itemCostPrice;

    /**
     * Set itemCostPrice
     *
     * @param float $itemCostPricePrice
     * @return OrderItem
     */
    public function setItemCostPrice($itemCostPrice)
    {
        $this->itemCostPrice = $itemCostPrice;
        return $this;
    }

    /**
     * Get itemCostPrice
     *
     * @return float
     */
    public function getItemCostPrice()
    {
        return $this->itemCostPrice;
    }

    /**
     * Renew order item
     *
     * @return boolean
     */
    public function renew()
    {
        $result = parent::renew();

        if ($result) {
            $this->setItemCostPrice($this->getProduct()->getCostPrice());
        }

        return $result;
    }

    /**
     * Get deleted product
     *
     * @return \XLite\Model\Product|void
     */
    protected function getDeletedProduct()
    {
        $dumpProduct = parent::getDeletedProduct();

        $dumpProduct->setCostPrice($this->getItemCostPrice());

        return $dumpProduct;
    }

    /**
     * Save item state
     *
     * @param \XLite\Model\Base\IOrderItem $item Item object
     *
     * @return void
     */
    protected function saveItemState(\XLite\Model\Base\IOrderItem $item)
    {
        parent::saveItemState($item);

        $this->setItemCostPrice($item->getCostPrice());
    }

    /**
     * Reset item state
     *
     * @return void
     */
    protected function resetItemState()
    {
        parent::resetItemState();

        $this->itemCostPrice = 0;
    }

    /**
     * Get clear cost price value
     *
     * @return float
     */
    public function getProductCostPrice()
    {
        return $this->getProduct()->getCostPrice();
    }

    /**
     * Get use for cost price
     * @return boolean
     */
    public function isUseCostPrice()
    {
        return $this->isOrderOpen() && !$this->isDeleted();
    }

    /**
     * Get clear cost price value
     *
     * @return float
     */
    public function getCostPrice()
    {
        return $this->isUseCostPrice() ? $this->getProductCostPrice() : $this->getItemCostPrice();
    }

    /**
     * Calculate  Cost Price subtotal
     *
     * @return float
     */
    public function calculateCostPriceSubtotal()
    {
        if ($this->isOrderOpen() || $this->getCostPrice() === null) {
            $this->setItemCostPrice($this->getProductCostPrice());
        }

        return $this->getCostPrice() * $this->getAmount();
    }

    public function calculate()
    {
        parent::calculate();

        $this->calculateCostPriceSubtotal();
    }
}
