<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\CostPrice\Controller\Admin;

use XCart\Extender\Mapping\Extender;

/**
 * Orders statistics page controller
 * @Extender\Mixin
 */
class OrdersStats extends \XLite\Controller\Admin\OrdersStats
{
    /**
     * Columns
     */
    public const P_GROSS_PROFIT = 'grossProfit';

    protected $grossProfitInitialized = false;

    /**
     * Get row headings
     *
     * @return array
     */
    public function getRowTitles()
    {
        return array_slice(parent::getRowTitles(), 0, 5) + [
                self::P_GROSS_PROFIT => static::t('Gross profit'),
            ] + array_slice(parent::getRowTitles(), 5);
    }

    /**
     * Status rows as row identificator => included statuses
     *
     * @return array
     */
    public function getStatusRows()
    {
        return array_slice(parent::getStatusRows(), 0, 5) + [
                static::P_GROSS_PROFIT => [
                    \XLite\Model\Order\Status\Payment::STATUS_AUTHORIZED,
                    \XLite\Model\Order\Status\Payment::STATUS_PAID,
                    \XLite\Model\Order\Status\Payment::STATUS_PART_PAID,
                ],
            ] + array_slice(parent::getStatusRows(), 5);
    }

    /**
     * Is Gross Profit row
     *
     * @param string $row Row identificator
     *
     * @return boolean
     */
    public function isGrossProfitRow($row)
    {
        return in_array(
            $row,
            [
                self::P_GROSS_PROFIT,
            ]
        );
    }

    /**
     * Is totals row
     *
     * @param string $row Row identificator
     *
     * @return boolean
     */
    public function isTotalsRow($row)
    {
        return parent::isTotalsRow($row) || $row === self::P_GROSS_PROFIT;
    }

    /**
     * Get gross data
     *
     * @param integer $startTime Start time
     *
     * @return array
     */
    protected function getGrossData($startTime)
    {
        $condition = $this->defineGetDataTotalCondition($startTime);

        $condition->codes = [
            \XLite\Model\Order\Status\Payment::STATUS_AUTHORIZED,
            \XLite\Model\Order\Status\Payment::STATUS_PAID,
            \XLite\Model\Order\Status\Payment::STATUS_PART_PAID,
        ];

        return \XLite\Core\Database::getRepo('XLite\Model\Order')->getStatisticGrossProfit($condition);
    }

    public function getStats()
    {
        parent::getStats();

        if (!$this->grossProfitInitialized) {
            foreach ($this->stats[self::P_GROSS_PROFIT] as $period => $_v) {
                $this->stats[self::P_GROSS_PROFIT][$period] = $this->getGrossData($this->getStartTime($period));
            }

            $this->grossProfitInitialized = true;
        }

        return parent::getStats();
    }
}
