<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\CloudSearch\Core;

use QSL\Make\Main as MakeMain;
use QSL\Make\Model\Product as MakeProduct;
use XCart\Extender\Mapping\Extender;
use XLite\Core\Database;
use XLite\Model\Product;

/**
 * CloudSearch store-side API methods
 *
 * @Extender\Mixin
 * @Extender\Depend ({"QSL\Make"})
 */
class StoreApiMMY extends \QSL\CloudSearch\Core\StoreApi
{
    /**
     * Get "conditions" that can be used to restrict the results when searching.
     *
     * This is different from "attributes" which are used to construct full-fledged filters (CloudFilters).
     *
     * @param Product $product
     *
     * @return array
     */
    protected function getProductConditions(Product $product)
    {
        $conditions = parent::getProductConditions($product);

        $conditions['mmy'] = [];

        $mapping = [
            MakeProduct::UNIVERSAL_FIT   => 100,
            MakeProduct::REGULAR_PRODUCT => 10,
        ];

        if (isset($mapping[$product->getFitmentType()])) {
            $conditions['mmy'][] = $mapping[$product->getFitmentType()];
        }

        return $conditions;
    }

    /**
     * Get sort fields that can be used to sort CloudSearch search results.
     * Sort fields are dynamic in the way that custom sort_int_*, sort_float_*, sort_str_* are allowed.
     *
     * @param Product $product
     *
     * @return array
     */
    protected function getSortFields(Product $product)
    {
        $mapping = [
            MakeProduct::VEHICLE_SPECIFIC => 1000,
            MakeProduct::UNIVERSAL_FIT    => 100,
            MakeProduct::REGULAR_PRODUCT  => 10,
        ];

        return parent::getSortFields($product) + ['sort_int_fitment_type' => $mapping[$product->getFitmentType()]];
    }

    protected function enrichProductsData(&$products): void
    {
        parent::enrichProductsData($products);

        if (empty($products)) {
            return;
        }

        $qb = Database::getEM()->createQueryBuilder();
        $results = $qb
            ->select('IDENTITY(l.product) AS productId')
            ->addSelect("GROUP_CONCAT(IDENTITY(l.level) SEPARATOR ',') AS levelIds")
            ->from(MakeMain::getLastLevelProductRepository(), 'l')
            ->where($qb->expr()->in('l.product', array_keys($products)))
            ->groupBy('l.product')
            ->getQuery()
            ->getArrayResult();

        if (!empty($results)) {
            foreach ($results as $result) {
                $products[$result['productId']]['conditions']['mmy'] = array_map(
                    static fn($l) => "level_$l",
                    explode(',', $result['levelIds'])
                );
            }
        }
    }
}
