<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\CloudSearch\Core;

use Includes\Utils\ConfigParser;
use Includes\Utils\URLManager;
use Symfony\Component\HttpClient\HttpClient;
use Symfony\Contracts\HttpClient\Exception\ExceptionInterface;
use Symfony\Contracts\HttpClient\HttpClientInterface;
use XLite\Core\Config;
use XLite\Core\Database;
use XLite\Core\Router;
use XLite\Core\Session;
use XLite\InjectLoggerTrait;
use QSL\CloudSearch\Main;

/**
 * CloudSearch API client
 */
class ServiceApiClient
{
    use InjectLoggerTrait;

    /**
     * CloudSearch service access details
     */
    public const CLOUD_SEARCH_URL               = 'https://cloudsearch.x-cart.com';
    public const CLOUD_SEARCH_REMOTE_IFRAME_URL = '/api/v1/iframe?key=';
    public const CLOUD_SEARCH_REGISTER_URL      = '/api/v1/register';
    public const CLOUD_SEARCH_SEARCH_URL        = '/api/v1/search';
    public const CLOUD_SEARCH_PLAN_INFO_URL     = '/api/v1/plan-info';
    public const CLOUD_SEARCH_WEBHOOK_URL       = '/api/v1/webhook';

    public const SEARCH_REQUEST_TIMEOUT    = 5;
    public const PLAN_INFO_REQUEST_TIMEOUT = 3;

    public const WEBHOOK_TIMEOUT = 3;

    protected static array $resultsCache = [];

    protected function getHttpClient(): HttpClientInterface
    {
        return HttpClient::create();
    }

    /**
     * Request CloudSearch registration
     */
    public function register(): void
    {
        $requestUrl = $this->getCloudSearchUrl() . static::CLOUD_SEARCH_REGISTER_URL;

        $shopUrl = $this->getShopUrl();

        $shopKey = md5(uniqid(rand(), true));

        $requestBody = [
            'shopUrl'  => $shopUrl,
            'shopKey'  => $shopKey,
            'shopType' => 'xc5',
            'extra'    => 'symfony',
        ];

        Database::getRepo('XLite\Model\TmpVar')->setVar('cloud_search_shop_key', $shopKey);

        try {
            $response = $this->getHttpClient()->request(
                'POST',
                $requestUrl,
                [
                    'body' => $requestBody,
                ]
            );

            if ($response->getStatusCode() === 200) {
                $data = json_decode($response->getContent(), true);

                if ($data && !empty($data['apiKey'])) {
                    $this->storeApiKey($data['apiKey']);

                    Config::updateInstance();
                }

                Database::getRepo('XLite\Model\TmpVar')->removeVar('cloud_search_shop_key');
            }
        } catch (ExceptionInterface $e) {
            $this->getLogger('CloudSearchLogs')->error('API Error', [
                'requestUrl'       => $requestUrl,
                'requestBody'      => $requestBody,
                'exceptionMessage' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Search functionality on the product list
     */
    public function search(SearchParametersInterface $params): ?array
    {
        $params = $params->getParameters();

        $paramsHash = md5(serialize($params));

        if (!array_key_exists($paramsHash, self::$resultsCache)) {
            $responseBody = $this->performSearchRequest($params);

            self::$resultsCache[$paramsHash] = $responseBody
                ? $this->extractSearchResultsFromResponse($responseBody)
                : null;
        }

        return self::$resultsCache[$paramsHash];
    }

    /**
     * Get CloudSearch service URL that defaults to https://cloudsearch.x-cart.com but can be overridden
     * with CLOUD_SEARCH_URL env var
     */
    protected function getCloudSearchUrl(): string
    {
        return !empty($_SERVER['CLOUD_SEARCH_URL']) ? $_SERVER['CLOUD_SEARCH_URL'] : static::CLOUD_SEARCH_URL;
    }

    /**
     * Get search api endpoint url
     */
    public function getSearchApiUrl(): string
    {
        return $this->getCloudSearchUrl() . static::CLOUD_SEARCH_SEARCH_URL;
    }

    /**
     * Get CloudSearch API key
     */
    public function getApiKey(): mixed
    {
        return Config::getInstance()->QSL->CloudSearch->api_key;
    }

    /**
     * Get CloudSearch API key
     */
    public function getSecretKey(): mixed
    {
        return Config::getInstance()->QSL->CloudSearch->secret_key;
    }

    /**
     * Retrieve search results from the response body
     */
    protected function extractSearchResultsFromResponse(array $input): array
    {
        $products = $input
            && $input['products']
            && count($input['products']) > 0 ? $input['products'] : [];

        return [
            'products'         => $products,
            'numFoundProducts' => $input['numFoundProducts'],
            'facets'           => $input['facets'],
            'stats'            => $input['stats'],
        ];
    }

    /**
     * Perform product search request (ALL) into the CloudSearch service
     */
    protected function performSearchRequest(array $params): ?array
    {
        $requestUrl = $this->getSearchApiUrl();
        $data       = [
                'apiKey' => $this->getApiKey(),
                'all'    => 1,
            ] + $params;

        try {
            $response = $this->getHttpClient()->request(
                'POST',
                $requestUrl,
                [
                    'max_duration' => self::SEARCH_REQUEST_TIMEOUT,
                    'json'         => $data,
                ]
            );

            if ($response->getStatusCode() === 200) {
                return json_decode($response->getContent(), true);
            }
        } catch (ExceptionInterface $e) {
            $this->getLogger('CloudSearchLogs')->error('API Error', [
                'requestUrl'       => $requestUrl,
                'requestBody'      => $data,
                'exceptionMessage' => $e->getMessage(),
            ]);
        }

        return null;
    }

    /**
     * Request CS plan info
     */
    public function getPlanInfo(): mixed
    {
        $apiKey      = $this->getApiKey();
        $secretKey   = $this->getSecretKey();
        $requestUrl  = $this->getCloudSearchUrl() . static::CLOUD_SEARCH_PLAN_INFO_URL;
        $requestBody = ['apiKey' => $apiKey, 'secretKey' => $secretKey];

        try {
            $response = $this->getHttpClient()->request(
                'POST',
                $requestUrl,
                [
                    'body'         => $requestBody,
                    'max_duration' => self::PLAN_INFO_REQUEST_TIMEOUT,
                ]
            );

            if ($response->getStatusCode() === 200) {
                return json_decode($response->getContent(), true);
            }
        } catch (ExceptionInterface $e) {
            $this->getLogger('CloudSearchLogs')->error('API Error', [
                'requestUrl'       => $requestUrl,
                'requestBody'      => $requestBody,
                'exceptionMessage' => $e->getMessage(),
            ]);
        }

        return null;
    }

    /**
     * Get CloudSearch dashboard url
     */
    public function getDashboardIframeUrl(string $secretKey, array $params): string
    {
        $features = ['cloud_filters', 'admin_search'];

        return $this->getCloudSearchUrl()
            . static::CLOUD_SEARCH_REMOTE_IFRAME_URL
            . $secretKey
            . '&' . http_build_query($params +
                [
                    'client_features' => $features,
                    'locale'          => Session::getInstance()->getLanguage()->getCode(),
                ]);
    }

    /**
     * Get store url without script part
     */
    protected function getShopUrl(): string
    {
        $router = Router::getInstance();

        if (method_exists($router, 'disableLanguageUrlsTmp')) {
            $router->disableLanguageUrlsTmp();
        }

        $url = URLManager::getShopURL();

        if (method_exists($router, 'releaseLanguageUrlsTmp')) {
            $router->releaseLanguageUrlsTmp();
        }

        $protocol = URLManager::isHTTPS() ? 'https' : 'http';

        $hostDetails = ConfigParser::getOptions(['host_details']);

        if (Main::isMultiDomain() && isset($hostDetails[$protocol . '_host_orig'])) {
            $original_host = $hostDetails[$protocol . '_host_orig'];

            $scheme = parse_url($url, PHP_URL_SCHEME);
            $host   = parse_url($url, PHP_URL_HOST);

            $url = $scheme . '://' . $original_host
                . substr($url, strlen($scheme) + strlen('://') + strlen($host));
        }

        return $url;
    }

    /**
     * Store API key in the DB
     */
    protected function storeApiKey(string $key): void
    {
        Database::getRepo('XLite\Model\Config')->createOption(
            [
                'name'     => 'api_key',
                'category' => 'QSL\CloudSearch',
                'value'    => $key,
            ]
        );
    }

    public function sendWebhookEvent($eventData): void
    {
        $requestUrl = $this->getCloudSearchUrl() . static::CLOUD_SEARCH_WEBHOOK_URL;

        try {
            $response = $this->getHttpClient()->request(
                'POST',
                $requestUrl,
                [
                    'max_duration' => self::WEBHOOK_TIMEOUT,
                    'headers'      => [
                        'X-Api-Key' => $this->getSecretKey(),
                    ],
                    'json'         => $eventData,
                ]
            );

            $response->getHeaders();
        } catch (ExceptionInterface $e) {
            $this->getLogger('CloudSearchLogs')->error('Webhook error', [
                'requestUrl'       => $requestUrl,
                'requestBody'      => $eventData,
                'exceptionMessage' => $e->getMessage(),
            ]);
        }
    }
}
