<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\CloudSearch\Core\IndexingEvent;

use Doctrine\Common\EventSubscriber;
use Doctrine\ORM\Event\PreRemoveEventArgs;
use Doctrine\ORM\Event\PreUpdateEventArgs;
use Doctrine\ORM\Event\PostFlushEventArgs;
use Doctrine\ORM\Event\PostPersistEventArgs;
use Doctrine\ORM\Events;
use XLite\Core\Database;
use XLite\Model\Category;
use XLite\Model\CategoryProducts;
use XLite\Model\Image\Category\Image as CategoryImage;
use XLite\Model\Image\Product\Image as ProductImage;
use XLite\Model\Product;
use QSL\CloudSearch\Core\ServiceApiClient;

class IndexingEventListener implements EventSubscriber
{
    protected static $events = [];

    protected static $importEvents = [];

    protected static $isBatchMode = false;

    public static function isBatchMode(): bool
    {
        return self::$isBatchMode;
    }

    public static function setBatchMode(bool $flag = true): void
    {
        self::$isBatchMode = $flag;
    }

    public static function getImportStarted()
    {
        return Database::getRepo('XLite\Model\TmpVar')->getVar('csImportStarted');
    }

    public static function setImportStarted(bool $flag = true): void
    {
        $tmpVar = Database::getRepo('XLite\Model\TmpVar');
        if ($flag) {
            $tmpVar->setVar('csImportStarted', time());
        } else {
            $tmpVar->removeVar('csImportStarted');
        }
    }

    /**
     * Returns an array of events this subscriber wants to listen to.
     *
     * @return array
     */
    public function getSubscribedEvents()
    {
        return [
            Events::postPersist,
            Events::preUpdate,
            Events::preRemove,
            Events::postFlush,
        ];
    }

    public function postPersist(PostPersistEventArgs $eventArgs)
    {
        $startTime = microtime(true);

        $instance = $eventArgs->getObject();

        if ($instance instanceof IndexingEventTriggerInterface) {
            $ids = $instance->getCloudSearchEntityIds();

            if ($ids) {
                $action = $instance->getCloudSearchEventAction()
                    ?: IndexingEventTriggerInterface::INDEXING_EVENT_CREATED_ACTION;

                foreach ($ids as $id) {
                    $this->addEvent($id, $instance->getCloudSearchEntityType(), $action);
                }
            }
        }

        IndexingEventProfiler::getInstance()->addToTotalTime(microtime(true) - $startTime);
    }

    protected function getCloudSearchProductUpdateEventAction($instance, PreUpdateEventArgs $eventArgs)
    {
        if ($eventArgs->hasChangedField('enabled')) {
            return $instance->getEnabled()
                ? IndexingEventTriggerInterface::INDEXING_EVENT_CREATED_ACTION
                : IndexingEventTriggerInterface::INDEXING_EVENT_DELETED_ACTION;
        }

        return IndexingEventTriggerInterface::INDEXING_EVENT_UPDATED_ACTION;
    }

    public function preUpdate(PreUpdateEventArgs $eventArgs)
    {
        $startTime = microtime(true);

        $instance = $eventArgs->getObject();

        if ($instance instanceof IndexingEventTriggerInterface && $this->hasChanges($instance, $eventArgs)) {
            $ids = $instance->getCloudSearchEntityIds();

            if ($ids) {
                $action = $instance->getCloudSearchEventAction()
                    ?: $this->getCloudSearchProductUpdateEventAction($instance, $eventArgs);

                foreach ($ids as $id) {
                    $this->addEvent($id, $instance->getCloudSearchEntityType(), $action);
                }
            }
        }

        IndexingEventProfiler::getInstance()->addToTotalTime(microtime(true) - $startTime);
    }

    protected function hasChanges($instance, PreUpdateEventArgs $eventArgs)
    {
        if (
            $instance instanceof Product
            && (
                count($instance->getMemberships()->getDeleteDiff())
                || count($instance->getMemberships()->getInsertDiff())
            )
        ) {
            return true;
        }

        $changeSet = array_filter($eventArgs->getEntityChangeSet(), static function ($c) {
            return !is_scalar($c[0]) || !is_scalar($c[1]) || $c[0] != $c[1];
        });

        if (
            empty($changeSet)
            || $instance instanceof CategoryImage
            || $instance instanceof ProductImage
            || (
                $instance instanceof Product
                && array_intersect($this->getProductModelTrackFields(), array_keys($changeSet)) === []
            )
            || (
                $instance instanceof Category
                && array_intersect($this->getCategoryModelTrackFields(), array_keys($changeSet)) === []
            )
            || (
                $instance instanceof CategoryProducts
                && isset($changeSet['orderbyInProduct']) && count($changeSet) === 1
            )
        ) {
            return false;
        }

        return true;
    }

    protected function getProductModelTrackFields()
    {
        return [
            'sku',
            'price',
            'enabled',
            'amount',
            'vendor',
            'salePriceValue',
            'participateSale',
            'inventoryEnabled',
        ];
    }

    protected function getCategoryModelTrackFields()
    {
        return [
            'enabled',
            'parent',
        ];
    }

    public function preRemove(PreRemoveEventArgs $eventArgs)
    {
        $startTime = microtime(true);

        $instance = $eventArgs->getObject();

        if ($instance instanceof IndexingEventTriggerInterface) {
            $ids = $instance->getCloudSearchEntityIds();

            if ($ids) {
                $action = $instance->getCloudSearchEventAction()
                    ?: IndexingEventTriggerInterface::INDEXING_EVENT_DELETED_ACTION;

                foreach ($ids as $id) {
                    $this->addEvent($id, $instance->getCloudSearchEntityType(), $action);
                }
            }
        }

        IndexingEventProfiler::getInstance()->addToTotalTime(microtime(true) - $startTime);
    }

    public function postFlush(PostFlushEventArgs $args)
    {
        $startTime = microtime(true);

        if (!empty(static::$events)) {
            $apiClient = new ServiceApiClient();

            $sendStartTime = microtime(true);

            $apiClient->sendWebhookEvent(array_values(static::$events));

            IndexingEventProfiler::getInstance()->addToSendTime(microtime(true) - $sendStartTime);

            static::$events = [];
        }

        if (!empty(static::$importEvents)) {
            $this->updateCsLastUpdate();

            static::$importEvents = [];
        }

        IndexingEventProfiler::getInstance()->addToTotalTime(microtime(true) - $startTime);
    }

    protected function updateCsLastUpdate()
    {
        $updatedCategoryIds = $createdCategoryIds = [];
        $updatedProductIds = $createdProductIds = [];
        $timestamp = time();

        foreach (static::$importEvents as $e) {
            [$entityType, $action] = explode('.', $e['eventType']);

            if ($entityType === IndexingEventTriggerInterface::INDEXING_EVENT_PRODUCT_ENTITY) {
                if ($action === IndexingEventTriggerInterface::INDEXING_EVENT_CREATED_ACTION) {
                    $createdProductIds[] = $e['entityId'];
                } elseif ($action === IndexingEventTriggerInterface::INDEXING_EVENT_UPDATED_ACTION) {
                    $updatedProductIds[] = $e['entityId'];
                }
            }

            if ($entityType === IndexingEventTriggerInterface::INDEXING_EVENT_CATEGORY_ENTITY) {
                if ($action === IndexingEventTriggerInterface::INDEXING_EVENT_CREATED_ACTION) {
                    $createdCategoryIds[] = $e['entityId'];
                } elseif ($action === IndexingEventTriggerInterface::INDEXING_EVENT_UPDATED_ACTION) {
                    $updatedCategoryIds[] = $e['entityId'];
                }
            }
        }

        if ($updatedProductIds) {
            Database::getEM()
                ->createQuery(
                    'UPDATE XLite\Model\Product p SET p.csLastUpdate = :timestamp WHERE p.product_id IN (:ids)'
                )
                ->setParameter('timestamp', $timestamp)
                ->setParameter('ids', $updatedProductIds)
                ->execute();
        }

        if ($createdProductIds) {
            Database::getEM()
                ->createQuery(
                    'UPDATE XLite\Model\Product p SET p.csLastUpdate = :timestamp, p.csCreated = :timestamp WHERE p.product_id IN (:ids)'
                )
                ->setParameter('timestamp', $timestamp)
                ->setParameter('ids', $createdProductIds)
                ->execute();
        }

        if ($updatedCategoryIds) {
            Database::getEM()
                ->createQuery(
                    'UPDATE XLite\Model\Category c SET c.csLastUpdate = :timestamp WHERE c.category_id IN (:ids)'
                )
                ->setParameter('timestamp', $timestamp)
                ->setParameter('ids', $updatedCategoryIds)
                ->execute();
        }

        if ($createdCategoryIds) {
            Database::getEM()
                ->createQuery(
                    'UPDATE XLite\Model\Category c SET c.csLastUpdate = :timestamp, c.csCreated = :timestamp WHERE c.category_id IN (:ids)'
                )
                ->setParameter('timestamp', $timestamp)
                ->setParameter('ids', $createdCategoryIds)
                ->execute();
        }
    }

    protected function addEvent($id, $type, $action)
    {
        $key = $type . '_' . $id;

        if (
            (array_key_exists($key, static::$events) || array_key_exists($key, static::$importEvents))
            && $action !== IndexingEventTriggerInterface::INDEXING_EVENT_DELETED_ACTION
        ) {
            return;
        }

        if (
            self::isBatchMode()
            && in_array($type, [
                IndexingEventTriggerInterface::INDEXING_EVENT_PRODUCT_ENTITY,
                IndexingEventTriggerInterface::INDEXING_EVENT_CATEGORY_ENTITY
            ], true)
        ) {
            static::$importEvents[$key] = [
                'entityId'  => $id,
                'eventType' => $type . '.' . $action,
            ];
        } else {
            static::$events[$key] = [
                'entityId'  => $id,
                'eventType' => $type . '.' . $action,
            ];
        }
    }

    public static function triggerLatestChangesReindex()
    {
        $startTime = microtime(true);
        $updatedSince = self::getImportStarted();

        if (!$updatedSince) {
            return;
        }

        self::setImportStarted(false);

        $position = 0;

        do {
            $productEvents = Database::getEM()
                ->createQuery(
                    "SELECT p.product_id as entityId, CASE WHEN p.csCreated >= :timestamp THEN 'product.created' ELSE 'product.updated' END as eventType
                         FROM XLite\Model\Product p
                         WHERE p.csLastUpdate >= :timestamp"
                )
                ->setParameter('timestamp', $updatedSince)
                ->setFirstResult($position)
                ->setMaxResults(IndexingEventCore::MAX_RESULTS)
                ->getResult();

            if ($productEvents) {
                $apiClient = new ServiceApiClient();

                $sendStartTime = microtime(true);

                $apiClient->sendWebhookEvent($productEvents);

                IndexingEventProfiler::getInstance()->addToSendTime(microtime(true) - $sendStartTime);
            }

            $position += IndexingEventCore::MAX_RESULTS;
        } while (count($productEvents) === IndexingEventCore::MAX_RESULTS);

        $position = 0;

        do {
            $categoryEvents = Database::getEM()
                ->createQuery(
                    "SELECT c.category_id as entityId, CASE WHEN c.csCreated >= :timestamp THEN 'category.created' ELSE 'category.updated' END as eventType
                         FROM XLite\Model\Category c
                         WHERE c.csLastUpdate >= :timestamp"
                )
                ->setParameter('timestamp', $updatedSince)
                ->setFirstResult($position)
                ->setMaxResults(IndexingEventCore::MAX_RESULTS)
                ->getResult();

            if ($categoryEvents) {
                $apiClient = new ServiceApiClient();

                $sendStartTime = microtime(true);

                $apiClient->sendWebhookEvent($categoryEvents);

                IndexingEventProfiler::getInstance()->addToSendTime(microtime(true) - $sendStartTime);
            }

            $position += IndexingEventCore::MAX_RESULTS;
        } while (count($categoryEvents) === IndexingEventCore::MAX_RESULTS);

        IndexingEventProfiler::getInstance()->addToTotalTime(microtime(true) - $startTime);
    }
}
