<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\CloudSearch\Controller;

use XLite\Core\Config;
use XLite\Core\Database;
use XLite\Core\Request;
use QSL\CloudSearch\Core\ServiceApiClient;
use QSL\CloudSearch\Core\StoreApi;

/**
 * CloudSearch API implementation trait
 */
trait ApiControllerTrait
{
    /**
     * 'info' api verb
     *
     * @return void
     */
    protected function doActionInfo()
    {
        $api = StoreApi::getInstance();

        $data = $api->getApiSummary();

        $this->printJSONAndExit($data);
    }

    /**
     * 'profile' api verb
     *
     * @return void
     */
    protected function doActionProfile()
    {
        $client = new ServiceApiClient();

        if (Request::getInstance()->key !== $client->getSecretKey()) {
            header('HTTP/1.0 403 Forbidden', true, 403);
            die;
        }

        $config = Config::getInstance();

        $accountNames = [];
        $emails       = @unserialize($config->Company->site_administrator, ['allowed_classes' => false]);

        if (!$emails) {
            $emails = $config->Company->site_administrator ? [$config->Company->site_administrator] : [];
        }

        $profileRepo = Database::getRepo('XLite\Model\Profile');

        foreach ($emails as $email) {
            $profile = $profileRepo->findByLogin($email);

            $accountNames[] = $profile ? $profile->getName() : null;
        }

        $data = [
            'company_name'  => $config->Company->company_name,
            'emails'        => $emails,
            'account_names' => $accountNames,
            'timezone'      => $config->Units->time_zone,
        ];

        $this->printJSONAndExit($data);
    }

    /**
     * 'products' api verb
     *
     * @return void
     */
    protected function doActionProducts()
    {
        $api = StoreApi::getInstance();

        $data = $api->getProducts($this->getConditionParams());

        $this->printJSONAndExit($data);
    }

    /**
     * 'categories' api verb
     *
     * @return void
     */
    protected function doActionCategories()
    {
        $api = StoreApi::getInstance();

        $data = $api->getCategories($this->getConditionParams());

        $this->printJSONAndExit($data);
    }

    /**
     * 'pages' api verb
     *
     * @return void
     */
    protected function doActionPages()
    {
        $api = StoreApi::getInstance();

        [$start, $limit] = $this->getLimits();

        $data = $api->getPages($start, $limit);

        $this->printJSONAndExit($data);
    }

    protected function doActionManufacturers()
    {
        $api = StoreApi::getInstance();

        $data = $api->getBrands();

        $this->printJSONAndExit($data);
    }

    /**
     * Stores new secret key sent from CloudSearch server
     *
     * @return void
     */
    protected function doActionFinalizeRegistration()
    {
        $request = Request::getInstance();

        $shopKey = Database::getRepo('XLite\Model\TmpVar')->getVar('cloud_search_shop_key');

        if (empty($request->shopKey) || empty($request->key) || $shopKey !== $request->shopKey) {
            header('HTTP/1.0 403 Forbidden', true, 403);
            die;
        }

        $this->setSetting('secret_key', $request->key);

        $this->printJSONAndExit([]);
    }

    /**
     * Set plan features via API
     *
     * @return void
     */
    protected function doActionSetPlanFeatures()
    {
        $request = Request::getInstance();

        $apiClient = new ServiceApiClient();

        $secretKey = $apiClient->getSecretKey();

        if ($secretKey && $secretKey === $request->secret_key) {
            $this->setSetting('planFeatures', json_encode($request->features));
        }

        $this->printNoneAndExit();
    }

    /**
     * Change store settings via API
     *
     * @return void
     */
    protected function doActionSetSettings()
    {
        $request = Request::getInstance();

        $apiClient = new ServiceApiClient();

        $secretKey = $apiClient->getSecretKey();

        if ($secretKey && $secretKey === $request->secret_key) {
            $settings = $request->settings;

            if (isset($settings['isCloudFiltersEnabled'])) {
                $this->setSetting('isCloudFiltersEnabled', (bool) $settings['isCloudFiltersEnabled']);
            }

            if (isset($settings['isAdminSearchEnabled'])) {
                $this->setSetting('isAdminSearchEnabled', (bool) $settings['isAdminSearchEnabled']);
            }
        }

        $this->printNoneAndExit();
    }

    protected function setSetting($name, $value)
    {
        Database::getRepo('XLite\Model\Config')->createOption(
            [
                'name'     => $name,
                'category' => 'QSL\CloudSearch',
                'value'    => $value,
            ]
        );
    }

    protected function printJSONAndExit($data)
    {
        $this->silent = true;
        $this->setSuppressOutput(true);

        $content = json_encode($data);

        $xLite = \XLite::getInstance();
        $xLite->addHeader('Content-Type', 'application/json; charset=UTF-8');
        $xLite->addHeader('Content-Length', strlen($content));
        $xLite->setContent($content);
    }

    protected function printNoneAndExit()
    {
        $this->silent = true;
        $this->setSuppressOutput(true);
    }

    /**
     * @return array
     */
    protected function getConditionParams()
    {
        $params = [];

        $request = Request::getInstance();

        if (isset($request->ids)) {
            $params['ids'] = $request->ids;
        } else {
            $params['start']  = isset($request->start) ? intval($request->start) : 0;
            $params['cursor'] = isset($request->cursor) ? intval($request->cursor) : 0;
            $params['limit']  = max(1, isset($request->limit) ? intval($request->limit) : StoreApi::MAX_ENTITIES_AT_ONCE);
        }

        return $params;
    }

    /**
     * Get adjusted request limits
     *
     * @return array
     */
    protected function getLimits()
    {
        $request = Request::getInstance();

        $start = max(0, $request->start);
        $limit = max(1, $request->limit ?: StoreApi::MAX_ENTITIES_AT_ONCE);

        return [$start, $limit];
    }
}
