<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\View\ItemsList\Model;

use QSL\Backorder\Model\BackorderedProduct;

class BackorderedProducts extends \XLite\View\ItemsList\Model\Table
{
    /**
     * Sorting parameter values
     */
    public const SORT_BY_MODE_AMOUNT = 'amount';
    public const SORT_BY_MODE_DATE   = 'o.date';
    public const SORT_BY_MODE_NAME   = 'oi.name';
    public const SORT_BY_MODE_SKU    = 'oi.sku';

    /**
     * Define and set widget attributes; initialize widget
     *
     * @param array $params Widget params OPTIONAL
     */
    public function __construct(array $params = [])
    {
        $this->sortByModes += [
            self::SORT_BY_MODE_AMOUNT => 'Amount',
            self::SORT_BY_MODE_NAME   => 'Name',
            self::SORT_BY_MODE_SKU    => 'SKU',
        ];

        parent::__construct($params);
    }

    protected function defineColumns()
    {
        return [
            'sku'    => [
                static::COLUMN_NAME    => static::t('SKU'),
                static::COLUMN_SORT    => static::SORT_BY_MODE_SKU,
                static::COLUMN_ORDERBY => 100,
            ],
            'name'   => [
                static::COLUMN_NAME    => static::t('Name'),
                static::COLUMN_MAIN    => true,
                static::COLUMN_ORDERBY => 200,
                static::COLUMN_SORT    => static::SORT_BY_MODE_NAME,
            ],
            'amount' => [
                static::COLUMN_NAME    => static::t('Amount'),
                static::COLUMN_ORDERBY => 300,
                static::COLUMN_SORT    => static::SORT_BY_MODE_AMOUNT,
            ],
            'orders' => [
                static::COLUMN_NAME    => static::t('Orders'),
                static::COLUMN_ORDERBY => 400,
            ],
        ];
    }

    /**
     * getSortByModeDefault
     *
     * @return string
     */
    protected function getSortByModeDefault()
    {
        return static::SORT_BY_MODE_NAME;
    }

    /**
     * Return true if items list should be displayed in static mode (no editable widgets, no controls)
     *
     * @return boolean
     */
    protected function isStatic()
    {
        return true;
    }

    /**
     * Get URL common parameters
     *
     * @return array
     */
    protected function getCommonParams()
    {
        $this->commonParams         = parent::getCommonParams();
        $this->commonParams['page'] = \XLite\Core\Request::getInstance()->page;

        return $this->commonParams;
    }

    protected function removeCntFromArray($array)
    {
        $return = [];

        foreach ($array as $element) {
            $return[] = $element[0];
        }

        return $return;
    }

    protected function getData(\XLite\Core\CommonCell $cnd, $countOnly = false)
    {
        $cnd->{\XLite\Model\Repo\OrderItem::P_ORDER_BY} = $this->getOrderBy();

        return \XLite\Core\Database::getRepo('XLite\Model\OrderItem')->getBackorders($cnd, $countOnly);
    }

    /**
     * Get page data
     *
     * @return array
     */
    protected function getPageData()
    {
        $data = parent::getPageData();

        $return = [];

        foreach ($data as $entity) {
            $return[] = $this->defineBackorderedProduct($entity);
        }

        return $return;
    }

    protected function defineBackorderedProduct($dataArray)
    {
        $return = new BackorderedProduct();

        $return->setProductId($dataArray['product_id']);
        $return->setAmount($dataArray['amount']);

        return $return;
    }

    protected function getNameColumnValue(BackorderedProduct $product)
    {
        return '<a href="/admin/?target=product&product_id=' . $product->getProduct()->getProductId() . '">' . $product->getProduct()->getName() . '</a>';
    }

    protected function getAmountColumnValue(BackorderedProduct $product)
    {
        return $product->getAmount();
    }

    protected function getOrdersColumnValue(BackorderedProduct $product)
    {
        $return = '';

        $orderItems = $this->getOrderItems($product);

        foreach ($orderItems as $orderItem) {
            if (
                get_class($orderItem->getOrder()) === 'XLite\Model\Order'
                && $orderItem->getBackorderAmount() > $orderItem->getBackorderClosedAmount()
            ) {
                $orderNumber = $orderItem->getOrder()->getOrderNumber();

                if (!empty($orderNumber)) {
                    $printableOrderNumber = $orderItem->getOrder()->getPrintableOrderNumber();
                } else {
                    $printableOrderNumber = $orderItem->getOrder()->getParent()->getPrintableOrderNumber();
                    $orderNumber          = $orderItem->getOrder()->getParent()->getOrderNumber();
                }

                $return .= '<a href="/admin/?target=order&order_number=' . $orderNumber . '">' . $printableOrderNumber . '</a><br />';
            }
        }

        return rtrim($return);
    }

    protected function getOrderItems(BackorderedProduct $product)
    {
        return $product->getProduct()->getOrderItems();
    }

    protected function defineRepositoryName()
    {
        return 'XLite\Model\OrderItem';
    }

    protected function getPanelClass()
    {
        return 'QSL\Backorder\View\StickyPanel\ItemsList\BackorderProduct';
    }
}
