<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Module\XC\ProductVariants\View\ItemsList\Model;

use XCart\Extender\Mapping\Extender;

/**
 * @Extender\Mixin
 * @Extender\Depend ("XC\ProductVariants")
 */
class OrderItem extends \XLite\View\ItemsList\Model\OrderItem
{
    /**
     * Add 'backorderAmount' to the order items data fields list
     *
     * @return array
     */
    protected function getOrderItemsDataFields()
    {
        $result = parent::getOrderItemsDataFields();
        $result[] = 'backorderAmount';

        return $result;
    }

    /**
     * Change product/variant quantity in stock if needed
     *
     * @param \XLite\Model\OrderItem $entity Order item entity
     *
     * @return void
     */
    protected function changeItemAmountInStock($entity)
    {
        // override specific case:
        if ($entity->getVariant()) {
            $oldVariant = $this->orderItemsData[$entity->getItemId()]['variant'];
            $newVariant = $entity->getVariant();

            if ($this->isItemDataChangedVariant($oldVariant, $newVariant)) {
                $oldVariantAmount = $this->orderItemsData[$entity->getItemId()]['amount'];
                $newVariantAmount = $entity->getAmount();

                if ($oldVariant->getProduct()->getIsAvailableForBackorder()) {
                    // Return old variant amount to stock -- backorder-specific procedure
                    $wasBackordered = (int)$this->orderItemsData[$entity->getItemId()]['backorderAmount'];
                    $this->returnOldVariantToStock($entity, $oldVariant, $oldVariantAmount, $wasBackordered);
                } else {
                    // Return old variant amount to stock -- default procedure
                    if (!$oldVariant->getDefaultAmount()) {
                        $oldVariant->changeAmount($oldVariantAmount);
                    } else {
                        $entity->getProduct()->changeAmount($oldVariantAmount);
                    }
                }

                if ($newVariant->getProduct()->getIsAvailableForBackorder()) {
                    // Take new variant amount from stock -- backorder-specific procedure
                    $this->takeNewVariantFromStock($entity, $newVariant, $newVariantAmount);
                } else {
                    // Get new variant amount from stock -- default procedure
                    if (!$newVariant->getDefaultAmount()) {
                        $newVariant->changeAmount(-1 * $newVariantAmount);
                    } else {
                        $entity->getProduct()->changeAmount(-1 * $newVariantAmount);
                    }
                }

                return;
            }
        }

        parent::changeItemAmountInStock($entity);
    }


    protected function returnOldVariantToStock(\XLite\Model\OrderItem $orderItem, \XC\ProductVariants\Model\ProductVariant $variant, $amount, $wasBackordered)
    {
        $product = $variant->getProduct();
        $stockable = $variant->getDefaultAmount() ? $product : $variant;

        $inStockAmount = $newInStockAmount = $stockable->getAmount();
        $backorderAmount = $newBackorderAmount = $orderItem->getBackorderAmount();

        $returnToBackstockPart = $wasBackordered; // old item's backorderAmount
        $returnToStockPart = max($amount - $returnToBackstockPart, 0);

        $stockableName = $this->getStockableName($stockable);
        $message = "[Inventory] {$stockableName} returned to stock:";

        if ($returnToBackstockPart > 0) {
            $message .= " {$returnToBackstockPart} was on backorder;";

            // ... 1st summand decreases order item's backorderAmount
            $newBackorderAmount = $backorderAmount - $returnToBackstockPart;
            // ^^^ could be just 0 here
            $orderItem->setBackorderAmount($newBackorderAmount);

            // ... and increases product's backorderLimit
            if ($product->getIsBackorderLimit()) {
                $product->setBackorderLimit(
                    $product->getBackorderLimit() + $returnToBackstockPart
                );
            }
        }

        // ... 2nd summand increases product's stock
        if ($returnToStockPart > 0) {
            $message .= " {$returnToStockPart} added to Quantity in stock amount;";

            $newInStockAmount = $inStockAmount + $returnToStockPart;
            $stockable->changeAmount($returnToStockPart);
        }

        $orderId = $orderItem->getOrder()->getOrderId();
        \XLite\Core\OrderHistory::getInstance()->registerEvent(
            $orderId,
            \XLite\Core\OrderHistory::CODE_CHANGE_AMOUNT,
            $message,
            [
                'orderId'    => $orderId,
                'newInStock' => $newInStockAmount,
                'oldInStock' => $inStockAmount,
                'product'    => $stockableName,
                'qty'        => $amount,
            ]
        );
    }

    protected function takeNewVariantFromStock($orderItem, $variant, $amount)
    {
        $product = $variant->getProduct();
        $stockable = $variant->getDefaultAmount() ? $product : $variant;

        $inStockAmount = $newInStockAmount = $stockable->getAmount();

        $stockableName = $this->getStockableName($stockable);
        $message = "[Inventory] {$stockableName} taken from stock:";
        $orderId = $orderItem->getOrder()->getOrderId();

        // ... split delta on two summands: 1. up until product's current Qty in stock and 2. the remaining
        if ($product->canBeBackordered()) {
            $takeFromStockPart = min($amount, $inStockAmount);
            $takeFromBackstockPart = max($amount - $takeFromStockPart, 0);
        } else {
            $takeFromStockPart = $amount;
            $takeFromBackstockPart = 0;

            if ($amount > $inStockAmount) {
                \XLite\Core\OrderHistory::getInstance()->registerEvent(
                    $orderId,
                    \XLite\Core\OrderHistory::CODE_CHANGE_AMOUNT,
                    \XLite\Core\OrderHistory::TXT_UNABLE_RESERVE_AMOUNT,
                    [
                        'product' => $stockableName,
                        'qty' => $takeFromStockPart,
                    ]
                );

                return;
            }
        }

        if ($takeFromStockPart > 0) {
            $message .= " {$takeFromStockPart} is subtracted from Quantity in stock amount;";

            // ... 1st summand decreases current product quantity (until depletion)
            $newInStockAmount = $inStockAmount - $takeFromStockPart;
            $stockable->changeAmount(-1 * $takeFromStockPart);
        }

        if ($takeFromBackstockPart > 0) {
            $message .= " {$takeFromBackstockPart} is now backordered;";

            // ... 2nd summand decreases product's backorderLimit
            if ($product->getIsBackorderLimit()) {
                $newLimit = max(0, $product->getBackorderLimit() - $takeFromBackstockPart);
                $product->setBackorderLimit($newLimit);
            }

            // ... and increases order item's backorderAmount
            $newBackorderAmount = $takeFromBackstockPart;
            $orderItem->setBackorderAmount($newBackorderAmount);
        }

        \XLite\Core\OrderHistory::getInstance()->registerEvent(
            $orderId,
            \XLite\Core\OrderHistory::CODE_CHANGE_AMOUNT,
            $message,
            [
                'orderId'    => $orderId,
                'newInStock' => $newInStockAmount,
                'oldInStock' => $inStockAmount,
                'product'    => $stockableName,
                'qty'        => $amount,
            ]
        );
    }

    /**
     * Builds item name for using in the Order History notifications
     *
     * @todo for variants, show selected options instead of sku
     *
     * @param \XLite\Model\Product|\XC\ProductVariants\Model\ProductVariant $stockable
     *
     * @return  string
     */
    protected function getStockableName($stockable)
    {
        if ($stockable instanceof \XLite\Model\Product) {
            return $stockable->getName();
        }

        return "{$stockable->getProduct()->getName()} ({$this->getVariantAttributesStr($stockable)})";
    }

    protected function getVariantAttributesStr($variant)
    {
        $attrs = [];
        foreach ($variant->getValues() as $attributeValue) {
            if ($attributeValue->getAttribute()->isVariable($variant->getProduct())) {
                $attrs[] = "{$attributeValue->getAttribute()->getName()}: {$attributeValue->asString()}";
            }
        }

        return implode('; ', $attrs);
    }
}
