<?php

/**
 * Copyright (c) 2011-present Qualiteam software Ltd. All rights reserved.
 * See https://www.x-cart.com/license-agreement.html for license details.
 */

namespace QSL\Backorder\Module\XC\ProductVariants\Model;

use XCart\Extender\Mapping\Extender;
use XLite\Model\Cart;

/**
 * @Extender\Mixin
 * @Extender\Depend ("XC\ProductVariants")
 */
abstract class ProductVariant extends \XC\ProductVariants\Model\ProductVariant
{
    public function setAmount($amount)
    {
        $this->amount = max(0, (int) $amount);
        return $this;
    }

    public function isOutOfStockDefault()
    {
        /** @var ProductVariantsStockAvailabilityPolicy $availabilityPolicy */
        $availabilityPolicy = $this->getProduct()->getStockAvailabilityPolicy();

        return $availabilityPolicy->isVariantOutOfStockDefault(Cart::getInstance(), $this->getId());
    }

    /**
     * Alias: is all product items in cart
     *
     * @return boolean
     */
    public function isAllStockInCart()
    {
        $res = parent::isAllStockInCart();
        if (! $res) {
            return false;
        }

        $product = $this->getProduct();
        if ($product->isUnlimitedBackorder()) {
            return false;
        } elseif (
            $product->isLimitedBackorder()
            && $this->getAvailableBackorderAmount() > 0
        ) {
            return false;
        }

        return true;
    }

    public function getAvailableBackorderAmount()
    {
        $product = $this->getProduct();

        if ($product->isUnlimitedBackorder()) {
            return $product->getMaxPurchaseLimit();
        }

        if ($product->isLimitedBackorder()) {
            return $product->getAvailableBackorderAmount();
        }

        return 0;
    }

    /**
     * Return max possibly product amount available to add to cart
     *
     * @return integer
     */
    public function getMaxAmount()
    {
        return parent::getMaxAmount() + $this->getAvailableBackorderAmount();
    }

    /**
     * How many product items added to cart
     *
     * @return boolean
     */
    public function getItemsInCartMessage()
    {
        $res = parent::getItemsInCartMessage();

        $cartBackorderAmount = $this->getProduct()->getInCartBackorderAmount();
        if ($cartBackorderAmount) {
            $res .= '<br><i class="icon-pencil"></i> ' . self::t(
                'X items of this product will be backordered',
                [
                    'backordered' => $cartBackorderAmount,
                ]
            );
        }

        return $res;
    }
}
